# >>> WinBUGSlogs2HtmlSummary <<<
#
# This program was written by
# ---------------------------
# Patrick Blisle
# Division of Clinical Epidemiology
# McGill University Health Centre
# Montreal, Qc CANADA
# patrick.belisle@clinepi.mcgill.ca

# For instructions on how to use this pgm, please visit
# http://www.medicine.mcgill.ca/epidemiology/Joseph/PBelisle/WinBUGSlogs2HtmlSummary.html


our %Instr = (
  allowMultipleNodesPerTable => 1,
  arrow                      => "n/a",       # what you want $Instr{arrow0} to be changed for in html summary
  arrow0                     => "->",        # what can be found in mean/sd columns for exp(node)
  blank                      => "&nbsp;",
  GUInCol                    => 6,
  ListSizes                  => 1,           # change to 0 if you do not want for-loops constant sizes (as read from data files) to be listed after each WinBUGS output file location
  outdir                     => "summaries", # default output sub-directory (comment out if you want output files to be saved in same directory as input files rather than in a subdirectory)
  ReportWidth                => 0,           # boolean
  row0                       => 15           # 1st row for nodes list
             );




my %Default = (IncludeMCStats => 1,                     # comment out if you'd like MC-Error, Start and Sample node statistics to be excluded from summary by default
               legal          => 1,                     # change to 0 if html summary table is to be printed on letter paper
               out            => "WinBUGS-Summary.html" # default title for Html output file
              );

$Instr{legal} = $Default{legal};


my @COLORS = ("beige", "white"); # alternating colors for tables lines

our %Color = (bg   => "white",
              lbls => "darkkhaki",
              main => "#000066"
             );

$Color{filename} = $Color{main};

my %Width = (char   =>    8,
             col    =>   60,
             letter =>  900, # landscape max widths (in pixels)
             legal  => 1200);


# ------------ Do not edit below this line --------------------------------------------------------------

use strict;
use Win32;
use Win32::Shortcut;
use Tkx;

# ------------------------------------------------------------------------------
my $AddLinks2Models;
our $AllDeselected;
our $AllSelected;
my $AnyInstructionOnNodes2Include;
my $definedfiles4pdf;
our $IncludeMCStats;
my $IncludePath;
my $IsSARDSSrcFile;
our $LogFile;
our $MyStartTime;
my $NStats;
our $Out;
our $Outdir;
my $outfile;
our $SARDSDir;
our $showCrIImbalance;
my $StatHeader;
my $title;
my $title0;
our $Title;

my @files4pdf;
our @logfiles;
our @nonodestatistics;
my @rcode;
our @SEP;
my @StatsColumns;

our %chk;
my %chkindex;
our %Col;
our %constant;
my %dim;
our %dir;
our %DistinctivePath;
my %DoNotInclude;
my %DoNotIncludeStar;
our %filename;
our %isDataFile;
our %model;
our %monitoredLast;
our %My;
our %niter;
our %sizeInDataFile;
our %stats;
our %user;
# ------------------------------------------------------------------------------


our %Pkg = (version => "1.8", # Released July 2015
            www     => "http://www.medicine.mcgill.ca/epidemiology/Joseph/PBelisle/WinBUGSlogs2HtmlSummary.html"
           );


&OpenSTDERR($0, 1);

# ------------------------------------------------------------------------------

our ($THIS) = reverse Win32::GetFullPathName($0);
$THIS = $1 if $THIS =~ /(.*)\./;



my %Style = (
  "a.me",
          "text-decoration: none; color: $Color{filename}; font-weight: bold;",
  "a.me : hover",
          "text-decoration: underline; color: darkgoldenrod;",
  "a.model",
          "text-decoration: none; color: $Color{filename};",
  "a.model :hover",
          "text-decoration: underline; color: $Color{filename};",
  "div.filesdates",
          "font-size: 8pt; color: $Color{filename}; border-style: solid; border-color: $Color{main}; border-width: 2px 0 1px 0;",
  "div.title",
          "color: $Color{main}; font-size: 15pt; font-weight: bold; font-style: italic;",
  "div.timestamp",
          "font-size: 8pt; color: $Color{filename}; border-style: solid; border-color: $Color{main}; border-width: 1px 0 2px 0;",
  "div.warning",
          "font-size: 8pt; font-weight: bold; color: crimson;",
  "span.distinctivepath",
          "font-size: 8pt; color: darkgray; font-style: italic;",
  "td.header",
          "color: white; background-color: $Color{main}; font-size: 10pt; text-align: left; padding-left: 4px; font-weight: bold;",
  "td.filedate",
          "font-size: 8pt; text-align: left; color: $Color{filename};",
  "td.fileheader",
          "font-size: 8pt; font-style: italic; padding-left: 10px; padding-right: 5px; text-aling: right; color: $Color{lbls};  border-style: solid; border-color: $Color{lbls}; border-width: 2px 1px 0 0;",
  "td.filename",
          "font-size: 8pt; text-align: left; color: $Color{filename}; padding-left: 10px;",
  "td.niter",
          "font-size: 8pt; text-align: right; color: $Color{filename}; padding-left: 10px; padding-right: 5px;",
  "td.sizeinfo",
          "font-size: 8pt; text-align: left; color: gray; padding-left: 10px; font-style: italic;",
  "td.stat",
          "font-size: 8pt; text-align: right; padding-right: 4px; padding-left: 10px;",
  "td.statr",
          "font-size: 8pt; text-align: right; padding-right: 4px; padding-left: 10px; border-style: solid; border-color: $Color{main}; border-width: 0 1px 0 0;",
  "td.statheader0",
          "font-size: 8pt; text-align: right; padding-right: 4px; padding-left: 10px; color: $Color{lbls}; border-style: solid; border-color: $Color{main}; border-width: 0 1px 0 0;",
  "td.statheader",
          "font-size: 8pt; text-align: right; padding-right: 4px; padding-left: 10px; color: $Color{lbls}; font-style: italic;",
  "td.statheaderr",
          "font-size: 8pt; text-align: right; padding-right: 4px; padding-left: 10px; color: $Color{lbls}; border-style: solid; border-color: $Color{main}; border-width: 0 1px 0 0; font-style: italic;",
  "td.logf",
          "font-size: 8pt; text-align: right; padding-right: 4px; color: $Color{filename}; border-style: solid; border-color: $Color{main}; border-width: 0 1px 0 0; padding-left: 4px;"
           );


while (@ARGV)
{
  $_ = shift;
  next if /\.odc/i && !$'; # ignore .odc files

  if ($_ eq "-o")
  {
    $user{outfile} = shift;
    ($user{outfname}) = reverse Win32::GetFullPathName($user{outfile});
  }
  elsif ($_ eq "-path")
  {
    $IncludePath = 1;
  }
  elsif (-f)
  {
    ($Outdir) = Win32::GetFullPathName($_) unless $Outdir;

    if (lc substr($_, -4) eq ".lnk")
    {
      my $link = new Win32::Shortcut( );
      $link->Load($_);
      $_ = $link->{'Path'};
      $link->Close( );
    }

    my $arg = $_;

    if (&IsListOfFiles($arg))
    {
      my @tmp;
      ($Outdir, @tmp) = &ListOfFiles($arg);
      push(@logfiles, @tmp);
    }
    elsif (&IsSARDSSrcFile($arg) && !@ARGV)
    {
      # option of local interest only
      $IsSARDSSrcFile = 1;
      &ReadSARDSSrcFile($arg);
    }
    else
    {
      push(@logfiles, $arg);
    }
  }
  elsif (substr($_, 0, 1) eq "*")
  {
    @logfiles = (<${_}>);
  }
}


die "No valid file name given in arguments. Abort.\n" unless @logfiles;


# ------------------------------------------------------------------------------------------------------
# WinBUGS node stats column headers

our %NodeStatColumnHeader = (
  lcl     => "2.5%",
  mcError => "MC error",
  mean    => "mean",
  median  => "median",
  node    => "node",
  sample  => "sample",
  sd      => "sd",
  start   => "start",
  ucl     => "97.5%",
  width   => "width"
  );



my @WinBUGSColKeys = ("node", "mean", "sd", "mcError", "lcl", "median", "ucl", "start", "sample"); # in the same order as they appear in WinBUGS log files

my $k = 0;
foreach (@WinBUGSColKeys)
{
  $Col{$_} = $k++;
}


my @AllStats = (0..$#WinBUGSColKeys);


our @NIterStats = ("start", "sample");


# Specify node stats to include in html summary, in order you want them to be reported

my @Report = ("mean", "sd", "mcError", "median", "lcl", "ucl", @NIterStats);


# ------------------------------------------------------------------------------------------------------
# First read each logfile, amongst which some may not be actual log files,
# but rather a list of log files to include in the summary

my @tmp = @logfiles;

foreach my $logfile (@tmp)
{
  my @tmplogfiles;

  open(TMP, $logfile) || die "Cannot read file (1) $logfile\nSorry\n";
  while (<TMP>)
  {
    next unless /\S/;
    chomp;

    if (-f)
    {
      push(@tmplogfiles, $_);
    }
    else
    {
      undef @tmplogfiles;
      last;
    }
  }
  close TMP;

  push(@logfiles, @tmplogfiles);
}

# --------------------------------------------------------------------------
&DefinePathDistinctiveParts(@logfiles);


foreach my $logf (@logfiles)
{
  my $tmp = Win32::GetLongPathName($logf);
  $tmp =~ s{\\+}{/}g;
  $tmp =~ /(.*)\//;
  ($dir{$logf}, $filename{$logf}) = ($1, $');

  my ($seenNodeStats, @log, %usedAltname);

  open(LOG, $logf) || die "Cannot read WinBUGS log file $logf. Abort.\n";

  while (<LOG>)
  {
    push(@log, $_);

    if (/(check|data)\((.*)\)/)
    {
      my ($cmd, $file) = ($1, $2);

      if ($cmd eq "check")
      {
        $model{$logf} = $file;
      }
      elsif ($cmd eq "data")
      {
        $isDataFile{$logf}{$file} = 1;
      }

      next;
    }


    next unless /\t/;

    s/\&/\&amp\;/g;
    s/\</\&lt\;/g;
    s/\>/\&gt\;/g;


    next unless /\t\s*\S+.*\s\d+\s+\d+\s*(\S*)/;
    my $altname = $1;
    next if /node\s+mean/;


    s/\t/ /g;

    my @tmp;

    if (/.*\]\S*/)
    {
      my $tmp;
      ($tmp, $_) = ($&, $');
      $tmp =~ s/\s*(.*)/\1/g; # ltrim
      $tmp =~ s/\s*,\s*/,/g;
      @tmp = ($tmp, split);
    }
    else
    {
      @tmp = split;
    }


    my @stats = @tmp[@AllStats];
    my $node0 = $tmp[$Col{node}];
    my $node = $altname || $node0;
    $stats{$node}{$logf} = join("\t", @stats);
    $seenNodeStats = 1;

    if ($altname)
    {
      if ($usedAltname{$altname})
      {
        die "Alternate node name $altname used twice in log file $logf. Sorry.\n\t(Once for node $usedAltname{$altname} and once for node $node).\n";
      }
      else
      {
        $usedAltname{$altname} = $altname;
      }
    }


    foreach my $stat (@NIterStats)
    {
      $niter{$stat}{$logf}{$node} = $tmp[$Col{$stat}];
    }

    if ($node =~ /\[/)
    {
      $dim{$`} .= $& . $' . "\t";
    }
    else
    {
      $dim{$node} .= "\t";
    }
  }
  close LOG;


  next if $seenNodeStats;

  if ($log[0] =~ /nodes.*exclude\s*from\s*summary/i)
  {
    $AnyInstructionOnNodes2Include = 1;
    shift @log;

    foreach (@log)
    {
      chomp;
      s/\s+//g;
      next unless /\S/;

      if (substr($_, -1) eq "*")
      {
        chop;
        $DoNotIncludeStar{$_} = 1;
      }
      else
      {
        $DoNotInclude{$_} = 1;
      }
    }

    next;
  }


  # BDT output files do not have TABs on their nodes stats output lines: thus, when they are input to this pgm
  # they need a special treatment

  shift @log until ($log[0] =~ /node statistics/i && !$` && $' !~ /\S/) || !@log;

  unless (@log)
  {
    print STDERR "WARNING: No node statistics found in WinBUGS logfile $logf\n";
    push(@nonodestatistics, $logf);
  }

  shift @log;

  while (@log)
  {
    $_ = shift @log;
    last unless /\S/;

    my @tmp = split;
    my @stats = @tmp[@AllStats];
    my $node = $tmp[$Col{node}];
    $stats{$node}{$logf} = join("\t", @stats);

    if ($node =~ /\[/)
    {
      $dim{$`} .= $& . $' . "\t";
    }
    else
    {
      $dim{$node} .= "\t";
    }
  }
}

# ---

@logfiles = sort byFileNameConsideringNumbers @logfiles;
my @nodes    = sort SortNodes keys %stats;

$Out .= ".html" unless $Out =~ /\.html?/i && !$';

chdir($Outdir) if $Outdir;


# --- Prompt user with a GUI to select nodes to report -------------------------------------------

unless ($IsSARDSSrcFile)
{
  my ($chkCol, $chkRow, %candidature);

  if ($Instr{outdir})
  {
    unless (-f $Instr{outdir})
    {
      unless (-e _)
      {
        mkdir($Instr{outdir}, 0777) || die "Cannot mkdir $Instr{outdir}\nSorry\n";
      }

      chdir($Instr{outdir});

      $Outdir = Win32::GetCwd();
    }
  }


  my $k = 0;
  foreach my $node (@nodes)
  {
    $_ = $node;
    s/\[.*\]//g;
    $candidature{$_} ||= ++$k;
    $chkindex{$node} = $candidature{$_};
  }

  my @candidates = sort {$candidature{$a} <=> $candidature{$b}} keys %candidature;

  ($chkRow, $chkCol) = ($Instr{row0}, 0);

  $title0 = &TitleUsedInMostRecentWinBUGSSummaryFileSameDir($Outdir); # will also read list of nodes monitored in this most recent WBLog summary
  $title0 ||= "Please enter a title here";
  $Title  = $title0;


  Tkx::wm_title(".", "WinBUGSLogs2HtmlSummary: Summarizing WinBUGS log output node statistics");
  Tkx::ttk__frame(".c");
  Tkx::grid( ".c", -column => $Instr{GUInCol}, -row => 20, -sticky => "nwes");
  Tkx::grid_columnconfigure( ".", 0, -weight => 1);
  Tkx::grid_rowconfigure(".", 0, -weight => 1);

  Tkx::ttk__label(".cpagetitle", -text => "Page title:");
  Tkx::grid(".cpagetitle", -row => 1, -column => 1, -sticky => "w");

  Tkx::ttk__entry(".cmypagetitle", -width => 40, -textvariable => \$title);
  Tkx::grid(".cmypagetitle", -row => 1, -column => 2, -sticky => "w", -columnspan => $Instr{GUInCol}-2);
  $title = $Title;

  Tkx::ttk__radiobutton(".crbLegal",  -text => "Legal",  -variable => \$Instr{legal}, -value => 1);
  Tkx::grid(".crbLegal",  -row => 1, -column => $Instr{GUInCol}, -sticky => "w");

  Tkx::ttk__checkbutton(".cbAddLinks2Models", -text => "Add links to models", -variable => \$AddLinks2Models);
  Tkx::grid(".cbAddLinks2Models", -column => 1, -row => 2, -sticky => "w");
  $AddLinks2Models = 1;

  Tkx::ttk__radiobutton(".crbLetter", -text => "Letter", -variable => \$Instr{legal}, -value => 0);
  Tkx::grid(".crbLetter", -row => 2, -column => $Instr{GUInCol}, -sticky => "w");

  Tkx::ttk__label(".clbloutputfilename", -text => "Output filename:");
  Tkx::grid(".clbloutputfilename", -column => 1, -row => 3, -sticky => "w");

  Tkx::ttk__entry(".coutputfilename", -width => 40, -textvariable => \$outfile);
  Tkx::grid(".coutputfilename", -column => 2, -row => 3, -sticky => "w", -columnspan => $Instr{GUInCol}-2);
  $outfile = $user{outfname} || $Default{out};

  Tkx::ttk__label(".cfileextension2use", -text => "(use .htm or .html file extension)");
  Tkx::grid(".cfileextension2use", -column => 2, -row => 4, -sticky => "w", -columnspan => $Instr{GUInCol}-2);

  Tkx::ttk__checkbutton(".cincludemcstats", -text => "Include MC Error, Start and Sample node statistics in summary", -variable => \$IncludeMCStats);
  Tkx::grid(".cincludemcstats", -column => 2, -row => 5, -sticky => "w", -columnspan => $Instr{GUInCol}-2);
  $IncludeMCStats = $Default{IncludeMCStats};


  Tkx::ttk__checkbutton(".cselectall", -text => "Select all", -command => \&SelectAll, -variable => \$AllSelected);
  Tkx::grid(".cselectall", -column => 1, -row => $Instr{row0}-1, -sticky => "w");

  Tkx::ttk__checkbutton(".cdeselectall", -text => "Deselect all", -command => \&DeselectAll, -variable => \$AllDeselected);
  Tkx::grid(".cdeselectall", -column => 2, -row => $Instr{row0}-1, -sticky => "w");

  Tkx::ttk__button(".cok", -text => ">> Ok <<", -command => \&Bye);
  Tkx::grid(".cok", -column => $Instr{GUInCol}, -row => $Instr{row0}-2, -sticky => "w");


  #Jobs
  foreach (@candidates)
  {
    my $deselect;

    if ($AnyInstructionOnNodes2Include)
    {
      $deselect = 0;

      if ($DoNotInclude{$_})
      {
        $deselect = 1;
      }
      else
      {
        foreach my $tmp (keys %DoNotIncludeStar)
        {
          if (substr($_, 0, length($tmp)) eq $tmp)
          {
            $deselect = 1;
            last;
          }
        }
      }
    }
    else
    {
      $deselect = 1;

      my $tmp = $_;
      $tmp =~ s/\[.*//;
      $deselect = 0 if $monitoredLast{$tmp};
    }

    my $k = $candidature{$_};
    print STDERR "Calling dim prototype for _ = $_ and dim = $dim{$_}\n"; # ICI tempo
    $_ .= &DimPrototype($dim{$_});

    $chkCol++;
    ($chkRow, $chkCol) = ($chkRow+1, 1) if $chkCol > $Instr{GUInCol};

    Tkx::ttk__checkbutton(".cb$k", -text => $_, -variable => \$chk{$k});
    Tkx::grid(".cb$k", -column => $chkCol, -row => $chkRow, -sticky => "w");
    $chk{$k} = 1 - $deselect;
  }

  Tkx::MainLoop();
}
else
{
  $IncludeMCStats = 0;
}

# ------------------------------------------------------------------------------

chdir($Outdir);

undef $outfile unless $outfile =~ /\S/;
$Out = $outfile || $Default{out};

$Out .= ".html" unless $Out =~ /\.html?/i && !$';
$Out = Win32::GetFullPathName($Out);


unlink($Out);

# ------------------------------------------------------------------------------
our $ConstantNIter = &ConstantNIter();

@Report = &StatsReport(@Report);
my $AnyExtraColumn = 1 if $Report[$#Report] ne "ucl";

push(@Report, "width") if $Instr{ReportWidth};

while (@Report)
{
  $_ = shift @Report;

  my $class = @Report ? "statheader" : "statheaderr";
  $StatHeader .= "<td class=$class width=$Width{col}> $NodeStatColumnHeader{$_} </td>";

  push(@StatsColumns, $Col{$_}) if defined $Col{$_};
  $NStats++;
}


my $NOSTATS = "<td colspan=$NStats class=statr>$Instr{blank}</td>";


# ---------------------------------------------------------------------
my @ignoredNodes;
our @nodes2summarize;

unless ($IsSARDSSrcFile)
{
  foreach my $node (@nodes)
  {
    if ($chk{$chkindex{$node}})
    {
      push(@nodes2summarize, $node);
    }
    else
    {
      push(@ignoredNodes, $node);
    }
  }
}


die "No node selected. Abort.\n" unless @nodes2summarize;

print STDERR "Title=$Title\n";


undef $Title if $Title eq $title0;

my $WIDTH = $Instr{legal} ? $Width{legal} : $Width{letter};

# ------------------------------------------------------------------------------------------------
# Compute # of vars to be printed in each subtable
my $tablenvars;

my ($lmax) = sort {length($b) <=> length($a)} values %filename; # find longest name in log files
$lmax = length($lmax);                                          # to estimate 1st column length

my $Width0 = 10 + int($Width{char} * $lmax);


if ($Instr{allowMultipleNodesPerTable})
{
  my $tmp = $WIDTH - $Width0;     # room left for stats
  $tmp /= $NStats * $Width{col};
  $tablenvars = int($tmp);
}
else
{
  $tablenvars = 1;
}

$tablenvars ||= 1;

# ------------------------------------------------------------------------------------------------
my @html = ("<html>",
            "<head>",
            "<style type=\"text/css\">");

foreach (keys %Style)
{
  push(@html, "$_\{", "  " . $Style{$_}, "\}");
}

push(@html, "</style>");
push(@html, "</head>");

# --

push(@html, "<body bgcolor=$Color{bg}>");
push(@html, "<div class=title> $title </div>") if $title;


while (@nodes2summarize)
{
  my %islogf;
  my @tmpnodes = splice(@nodes2summarize, 0, $tablenvars);

  my $width = $Width0 + $NStats * $Width{col} * @tmpnodes;
  push(@html, "<br><table cellspacing=0 cellpadding=0 width=$width>");

  push(@html, "<tr bgcolor=$Color{main}><td width=$Width0>$Instr{blank}</td>");
  foreach my $node (@tmpnodes)
  {
    push(@html, "<td colspan=$NStats class=header> $node </td>");
  }
  push(@html, "</tr>");

  push(@html, "<tr>");
  push(@html, "<td class=statheader0> File </td>");
  push(@html, $StatHeader x @tmpnodes);
  push(@html, "</tr><tr></tr>");


  foreach my $node (@tmpnodes)
  {
    foreach (keys %{$stats{$node}})
    {
      $islogf{$_} = 1;
    }
  }

  my @logf = sort byFileNameConsideringNumbers keys %islogf;


  my $color = 0;
  foreach my $logf (@logf)
  {
    push(@html, "<tr bgcolor=$COLORS[$color]>");

    my $tmp = $IncludePath ? $dir{$logf} : undef;
    $tmp .= $filename{$logf};

    $tmp = "<a href=\"$model{$logf}\" class=model>$tmp</a>" if $model{$logf};


    if (%DistinctivePath)
    {
      my $longpath = Win32::GetLongPathName($logf);
      $longpath =~ s{\\+}{/}g;
      $longpath = $1 if $longpath =~ /(.*)\//;
      $tmp = "<span class=distinctivepath>" . $DistinctivePath{$longpath} . "::</span>&nbsp;" . $tmp;
    }

    push(@html, "<td class=logf>$tmp</td>");

    foreach my $node (@tmpnodes)
    {
      if (exists $stats{$node}{$logf})
      {
        my ($imbalance, $w);
        my @stats = split("\t", $stats{$node}{$logf});


        $w = $stats[$Col{ucl}] - $stats[$Col{lcl}] if $Instr{ReportWidth};

        if ($showCrIImbalance && $node =~ /b\./i && !$`)
        {
          unless ($node =~ /\.(sd|prec)/i && !$')
          {
            if ($stats[$Col{lcl}] >= 0)
            {
              $imbalance = 1;
            }
            elsif ($stats[$Col{ucl}] <= 0)
            {
              $imbalance = -1;
            }
            else
            {
              $imbalance = $stats[$Col{ucl}] / ($stats[$Col{ucl}] - $stats[$Col{lcl}]);
              $imbalance-- if $imbalance < 0.5;
            }

            $imbalance = sprintf("%4.1f", 100 * $imbalance);
          }
        }

        @stats = @stats[@StatsColumns];

        while (@stats)
        {
          $_ = shift @stats;
          $_ = $Instr{arrow} if $_ eq $Instr{arrow0};
          my $class = @stats  || $AnyExtraColumn ? "stat" : "statr";
          push(@html, "<td class=$class> $_ </td>");
        }


        push(@html, "<td class=statr> $w </td>") if $Instr{ReportWidth};

        if ($showCrIImbalance)
        {
          $imbalance = "&mdash;" unless defined $imbalance;
          push(@html, "<td class=statr> $imbalance </td>");
        }
      }
      else
      {
        push(@html, $NOSTATS);
      }
    }

    push(@html, "</tr>");
    $color = 1 - $color;
  }

  push(@html, "</table>");




}


push(@html, &FileDates());


if (@ignoredNodes)
{
  push(@html, "<!-- nodes not included in this summary: " . &IgnoredNodesList(@ignoredNodes) . " -->");
}


push(@html, "</body></html>");


chdir($SARDSDir) if $IsSARDSSrcFile;

open(HTML, ">$Out") || die "Cannot write output to file $Out. Abort.\n";
foreach (@html)
{
  print HTML $_ . "\n";
}
close HTML;

$Out = Win32::GetFullPathName($Out) if $IsSARDSSrcFile;




print STDERR "Done.\nOutput sent to file $Out\n";


# ----- End of code -------------------------------------------------------------------------------


sub Bye
{
  Tkx::destroy(".");
} # end of Bye




































sub SelectAll
{
  if ($AllSelected)
  {
    foreach (keys %chk)
    {
      $chk{$_} = 1;
    }

    $AllDeselected = 0;
  }
} # end of SelectAll


sub byFileNameConsideringNumbers()
{
  my ($res, %str);
               
  ($str{u}, $str{v}) = (lc $filename{$a}, lc $filename{$b});
  ($str{u}, $str{v}) = (lc $dir{$a}, lc $dir{$b}) if $str{u} eq $str{v};

  until ($res)
  {
    if ($str{u} =~ /\d+/ && $str{v} =~ /\d+/)
    {
      my (%left, %num);
      
      foreach ("u", "v")
      {
        ($left{$_}, $num{$_}, $str{$_}) = &StrNumStr($str{$_});
      }
      
      $res = lc $left{u} cmp lc $left{v} || $num{u} <=> $num{v};
    }
    else
    {
      $res = $a cmp $b;
    }
  }
  
  $res;
} # end of byFileNameConsideringNumbers


sub CommaFmt()
{
  my ($x) = @_;

  $x = "$1,$2$'" while $x =~ /(\d+)(\d{3})/;

  $x;
} # end of CommaFmt


sub ConstantNIter()
{
  # Global variables:
  # -----------------
  # @NIterStats
  # %constant
  # %niter

  my $res = 1;

  foreach my $stat (@NIterStats)
  {
    my @logfiles = keys %{$niter{$stat}};

    foreach my $logf (@logfiles)
    {
      my @nodes = keys %{$niter{$stat}{$logf}};

      my $node = shift @nodes;
      my $x = $niter{$stat}{$logf}{$node};

      foreach my $node (@nodes)
      {
        next if $x == $niter{$stat}{$logf}{$node};
        $res = 0;
        last;
      }

      last unless $res;
      $constant{$logf}{$stat} = $x;
    }

    last unless $res;
  }

  $res;
} # end of ConstantNIter


sub DefinePathDistinctiveParts()
{
  my (@files) = @_;
  my %IsPath;

  # Global:
  # -------
  # %DistinctivePath

  foreach (@files)
  {
    my $path = Win32::GetLongPathName($_);
    $path =~ s{\\+}{/}g;
    $path = $1 if $path =~ /(.*)\//;
    $IsPath{$path} = 1;
  }

  my @paths = keys %IsPath;
  my $m = @paths;

  if ($m > 1)
  {
    my @longPaths = @paths; # save a copy of unmodified paths

    # Remove common head subdirectories

    while ()
    {
      my ($lastLeft, @right);

      foreach (@paths)
      {
        if (/\//)
        {
          my ($left, $right) = ($`, $');
          push(@right, $right);

          if (defined $lastLeft)
          {
            if ($left ne $lastLeft)
            {
              $lastLeft = "/";
              last;
            }
          }
          else
          {
            $lastLeft = $left;
          }
        }
        else
        {
          my $right;
          ($lastLeft, $right) = ("/", $_);
          push(@right, $right);
          last;
        }
      }

      last if $lastLeft eq "/";
      @paths = @right;
    }


    # Remove common trailing subdirectories

    while ()
    {
      my ($lastRight, @left);

      foreach (@paths)
      {
        if (/(.*)\//)
        {
          my ($left, $right) = ($1, $');
          push(@left, $left);

          if (defined $lastRight)
          {
            if ($right ne $lastRight)
            {
              $lastRight = "/";
              last;
            }
          }
          else
          {
            $lastRight = $right;
          }
        }
        else
        {
          my $left;
          ($lastRight, $left) = ("/", $_);
          push(@left, $left);
          last;
        }
      }


      last if $lastRight eq "/";
      @paths = @left;
    }

    # Attach the distinctive path part to each long path

    foreach (@longPaths)
    {
      $DistinctivePath{$_} = shift @paths;
    }
  }
} # end of DefinePathDistinctiveParts


sub DimPrototype()
{
  my ($dims) = @_;
  my ($dim, @dim);

  if ($dims =~ /\S/)
  {
    chop $dims;
    my @brackets = split(/\t/, $dims);
    
    foreach (@brackets)
    {
      my $d = 0;
      
      if (/\[/)
      {
        s/\s+//g;
        $_ = substr($_, 1, -1);
        my @tmp = split(/\,/, $_);
        $d = @tmp;
        my $k = 0;

        foreach (@tmp)
        {
          $dim = -1 if /\D/;
          $dim[$k] = $_ if $_ > $dim[$k];
          $k++;
        }
      }

      if (defined $dim)
      {
        if ($dim != $d)
        {
          $dim = -1;
          last;
        }
      }
      else
      {
        $dim = $d;
      } 
    }
  }


  unless (defined $dim)
  {
    "";
  }
  elsif ($dim < 0)
  {
    "[*]"
  }
  elsif ($dim == 0)
  {
    "";
  }
  else
  {
    "[1:" . join(",1:", @dim) . "]";
  }
} # end of DimPrototype


sub DimsOrder()
{
  my ($a, $b) = @_;
  my $res;
  
  my @a = split(",", $a);
  my @b = split(",", $b);

  until ($res)
  {
    ($a, $b) = (shift @a, shift @b);
    $res = &StringsOrder($a, $b);

    last if $res;

    if (@a)
    {
      $res = 1 unless @b;
    }
    elsif (@b)
    {
      $res = -1;
    }
    else
    {
      $res = 0;
      last;
    }
  }

  $res;
} # end of DimsOrder


sub FileDate()
{
  my ($what, $file, $fmt) = @_;
  my (@tmp,
      $atleast, $date, $time);

  # This fct was not designed to be called directly by user

  @tmp = stat($file);

  $time = $tmp[$what];

  if ($what == 10 && $time > $tmp[9])
  {
    ($time, $atleast) = ($tmp[9], 1);
  }

  $date = &MyDate($time, $fmt);
  $date = "<=" . $date if $atleast;

  $date;
} # end of FileDate


sub FileDates()
{
 my ($anySizeInfo, @table, %isSizeVar, %readDataFile);

  # Global variables:
  # -----------------
  # $ConstantNIter
  # @logfiles
  # @NIterStats
  # %Instr
  # %isDataFile
  # %model
  # %sizeInDataFile
  
  undef %sizeInDataFile;

  my @html = ("<br>" x 4,
              "<div class=filesdates>Files dates</div><br>");

  # table header
  my @header = ("<table cellspacing=0 cellpadding=0>",
                "<tr><td></td>",
                "<td class=fileheader>File name</td>");

  if ($ConstantNIter)
  {
    foreach my $stat (@NIterStats)
    {
      push(@header, "<td class=fileheader>$stat</td>");
    }
  }

  push(@header, "<td class=fileheader>path</td>");


  if ($Instr{ListSizes})
  {
    my @models = &Unique(values %model);

    foreach my $model (@models)
    {
      my @vars = &LoopSizes($model);
      foreach my $var (@vars)
      {
        $isSizeVar{$model}{$var} = 1;
      }
    }

    foreach my $logfile (@logfiles)
    {
      foreach my $file (keys %{$isDataFile{$logfile}})
      {
        unless ($readDataFile{$file})
        {
          $readDataFile{$file} = 1;
          &ReadSizesInDataFile($file);
        }
      }
    }
  }


  foreach my $logfile (@logfiles)
  {
    push(@table, "<tr><td class=filedate>" . &LastModificationDate("$dir{$logfile}/$filename{$logfile}", 1) . "</td>");
    push(@table, "<td class=filename>$filename{$logfile}</td>");

    if ($ConstantNIter)
    {
      foreach my $stat (@NIterStats)
      {
        push(@table, &NIter($constant{$logfile}{$stat}));
      }
    }

    push(@table, "<td class=filename>$dir{$logfile}</td>");


    if ($Instr{ListSizes})
    {
      my %size;
      my $model = $model{$logfile};

      my @datafiles = keys %{$isDataFile{$logfile}};
      foreach my $datafile (@datafiles)
      {
        my @vars = keys %{$sizeInDataFile{$datafile}};
        foreach my $var (@vars)
        {
          $size{$var} = $sizeInDataFile{$datafile}{$var} if $isSizeVar{$model}{$var};
        }
      }

      my @vars = sort {lc $a cmp lc $b || $a cmp $b} keys %size;

      if (@vars)
      {
        my @sizes;

        foreach my $var (@vars)
        {
          my $size = $size{$var};
          $size = &CommaFmt($size) if $size >= 10000;
          push(@sizes, "$var = $size");
        }

        my $tmp = join(", ", @sizes);
        push(@table, "<td class=sizeinfo>$tmp</td>");
        $anySizeInfo = 1;
      }
    }

    push(@table, "</tr>");
  }


  push(@header, "<td class=fileheader>size info</td>") if $anySizeInfo;
  unshift(@table, @header, "</tr><tr><td><br></td></tr>");
  push(@table, "</table>");
  push(@html, @table, "<br>");


  if (@nonodestatistics)
  {
    push(@html, "<div class=warning>Warning</div>");
    my $s = $#nonodestatistics > 0 ? "s" : "";
    push(@html, "No node statistics were found in the file$s below:<br>");

    foreach (@nonodestatistics)
    {
      push(@html, "&nbsp;&nbsp;&nbsp;$_<br>");
    }
    push(@html, "<br>");
  }

  push(@html, "<div class=timestamp>This file was written by <a href=\"$Pkg{www}\" class=me>$THIS $Pkg{version}</a> on " . localtime() . "</div>");

  @html;
} # end of FileDates


sub IgnoredNodesList()
{
  my (@nodes) = @_;
  my @list;

  @nodes = sort {lc $b cmp lc $a} @nodes;

  foreach (@nodes)
  {
    s/\[.*//;

    unshift(@list, $_) if $_ ne $list[0];
  }

  join(" ", @list);  
} # end of IgnoredNodesList


sub IsListOfFiles()
{
  my ($path) = @_;
  my ($allfiles, $anyfile) = (1, 0);

  open(MYTMP, $path) || die "Cannot read (2) file $path\nSorry\n";
  while (<MYTMP>)
  {
    chomp;
    next unless /\S/;
  
    if (-f)
    {
      $anyfile = 1;
    }
    else
    {
      $allfiles = 0;
      last;
    }
  }
  close MYTMP;

  $anyfile && $allfiles ? 1 : 0;
} # end of IsListOfFiles


sub IsSARDSSrcFile()
{
  my ($file) = @_;

  my ($dir) = Win32::GetFullPathName($file);
  $dir =~ s{\\}{/}g;
  $dir .= "\n";
  $dir =~ s{/\n}{};
  chomp $dir;
  
  $dir eq $SARDSDir ? 1 : 0;
} # end of IsSARDSSrcFile


sub LastModificationDate()
{
  my ($file, $fmt) = @_;
  &FileDate(9, $file, $fmt);
} # end of LastModificationDate


sub ListOfFiles()
{
  my ($path) = @_;
  my ($dir, $usedefaultdir);

  open(TMP, $path) || die "Cannot read (3) file in $path\nSorry\n";
  my @lof = (<TMP>);
  close TMP;
  chomp @lof;

  foreach (@lof)
  {
    my ($tmpdir) = Win32::GetFullPathName($_);

    if (defined $dir)
    {
      $usedefaultdir = 1 if $tmpdir ne $dir;
    }
    else
    {
      $dir = $tmpdir;
    }
  }

  $dir = $ENV{TEMP} || $ENV{TMP} if $usedefaultdir;

  ($dir, @lof);
} # end of ListOfFiles


sub LoopSizes()
{
  my ($model) = @_;
  my %isSizeVar;
  
  open(TMP, $model);
  while (<TMP>)
  {
    s/#.*//;
    if (/for\s*\(\s*\S+\s+in\s+(\S+)\s*:\s*(\S+)\s*\)/)
    {
      my @tmp = ($1, $2);
      foreach (@tmp)
      {
        $isSizeVar{$_} = 1 unless /\d/ && !$`;
      }
    }
  }
  close TMP;
  
  sort {lc $a cmp lc $b || $a cmp $b} keys %isSizeVar;
} # end of LoopSizes


sub MyDate()
{
  my ($time, $fmt, $ndaysinfuture) = @_;
  my ($mm, $dd, $yy, %mm);

  # input time is in Perl time (seconds since Jan 1, 1970 00:00:00)

  $time += $ndaysinfuture*60*60*24;
  $time = localtime($time);


  ($mm, $dd, $yy) = ($1, $2, $3) if $time =~ /([A-Za-z]+)\s+(\d+).*\s(\d+)/;

  if ($fmt <= 1)
  {
    "$dd-$mm-$yy";
  }
  else
  {
    %mm = ("Jan", 1, "Feb", 2, "Mar", 3, "Apr",  4, "May",  5, "Jun",  6,
           "Jul", 7, "Aug", 8, "Sep", 9, "Oct", 10, "Nov", 11, "Dec", 12);
    $mm = $mm{$mm};
    $mm = "0$mm" if $mm < 10;
    $dd = "0$dd" if $dd < 10;

    if ($fmt == 2)
    {
      "$yy/$mm/$dd";
    }
    elsif ($fmt == 3)
    {
      "$yy$mm$dd";
    }
    elsif ($fmt == 4)
    {
      "$dd/$mm/$yy";
    }
  }
} # end of MyDate


sub NIter()
{
  my ($m) = @_;

  $m = &CommaFmt($m) if $m >= 10000;

  "<td class=niter>$m</td>";
} # end of NIter


sub OpenSTDERR()
{
  my ($callingpgm, $timestamp, $definegloballogfile) = @_;
  my ($logfile, $logdir);

  # Global variables defined:
  # ------------------------------------------------------- 
  # $LogFile (optionally, that is, if $definegloballogfile)
  # $MyStartTime


  $logfile = "$1.log" if $callingpgm =~ /(.*)\./;
  ($logdir, $logfile) = Win32::GetFullPathName($logfile);
  $logdir = $& . "log\\" if $logdir =~ /c\:\\users\\patrick\.belisle\\Home\\bin\\/i;
  $logfile = $logdir . $logfile;

  open(STDERR, ">$logfile") || die "Cannot write to log file $logfile. Abort.\n";
  print STDERR localtime() . "\n\n" if $timestamp;

  $MyStartTime = time; # defined for future use, as in fct ReportRunTime, for example

  $LogFile = $logfile if $definegloballogfile;
}


sub PurgeRCodeFromNotMonitoredNodes()
{
  my (@rcode) = @_;
  my @new;
  
  while (@rcode)
  {
    my $tmp = pop @rcode;
    
    if ($tmp =~ /wblog\.plotci\(/)
    {
      my $previous = pop @rcode;
      
      if ($previous =~ /c\(\s*\)/)
      {
        pop @rcode while $rcode[$#rcode] =~ /c\(\s*\)/;
      }
      else
      {
        unshift(@new, $previous, $tmp);
      }
    }
    else
    {
      unshift(@new, $tmp);
    }
  }
  
  @new;
} # end of PurgeRCodeFromNotMonitoredNodes


sub ReadSARDSSrcFile()
{
  my ($file) = @_;
  my ($dno, $outdir, $prov, $prov0, $srcdir);

  # Global variables
  # ==================
  # $Outdir
  # @logfiles
  # @nodes2summarize
  # %dir

  my ($tmp) = Win32::GetFullPathName($file);
  chdir($tmp);

  undef @logfiles;

  open(TMP, $file) || die "Cannot read SARDS src file $file\nSorry\n";
  while (<TMP>)
  {
    next unless /\s*\-(\w+)\s+/ && !$`;
    my ($option, $etc) = (lc $1, $');

    if ($option eq "p")
    {
      $prov0 = $etc;
      $prov0 =~ s/\s+$//; # rtrim
      $prov = lc $prov0;
    }
    elsif ($option eq "d")
    {
      $dno = $etc;
      $dno =~ s/\s+$//; # rtrim
    }
    elsif ($option eq "d0")
    {
      $etc =~ s/\s+$//; # rtrim
      $srcdir = Win32::GetFullPathName($etc) if -d $etc;
    }
    elsif ($option eq "d1")
    {
      $etc =~ s/\s+$//; # rtrim
      $outdir = Win32::GetFullPathName($etc) if -d $etc;
    }
    elsif ($option eq "m" || $option eq "n")
    {
      $etc =~ s/,/ /g;
      push(@nodes2summarize, split(" ", $etc));
    }
  }
  close TMP;

  die "No nodes to monitor.\nSorry.\n" unless @nodes2summarize;
  die "Province and disease not specified.\nSorry.\n" unless $prov && $dno;
  die "Src dir not defined (through -d0 option)\nSorry\n" unless $srcdir;
  die "Out dir not defined (through -d1 option)\nSorry\n" unless $outdir;

  chdir($srcdir);
  $Outdir = $outdir;

  my @candidatewblogfiles = (<*-WinBUGSlog.txt>);

  foreach (@candidatewblogfiles)
  {
    my ($rightdno, $rightprov);

    my @tmp = split("-", $_);
    pop @tmp;
    foreach $tmp (@tmp)
    {
      if ($tmp eq $dno)
      {
        $rightdno = 1;
      }
      else
      {
        $tmp = lc $tmp;
        $rightprov = 1 if $tmp eq $prov;
      }
    }

    if ($rightdno && $rightprov)
    {
      push(@logfiles, $_);
      $dir{$_} = $srcdir;
    }
  }

  ($Out, $Title) = ("../WinBUGS-Summary-$prov0-$dno.html", join("", $SEP[0], $prov0, $SEP[1], $dno, $SEP[2]));
} # end of ReadSARDSSrcFile


sub ReadSizesInDataFile()
{
  my ($file) = @_;
  
  # Global: %sizeInDataFile
  
  open(TMP, $file);
  while (my $tmp = <TMP>)
  {
    $tmp =~ s/#.*//;
    while ($tmp =~ /(\S+)\s*=\s*(\d+)/)
    {
      my ($value, $var);
      ($var, $value, $tmp) = ($1, $2, $');
      $var =~ s/.*[,\(]//g;
      $var =~ s/\s+//g;
      $sizeInDataFile{$file}{$var} = $value;
    }
  }
  close TMP;
} # end of ReadSizesInDataFile


sub SortNodes()
{
  my (%a, %b);

  ($a{name}, $b{name}) = ($a, $b);

  $a{name} =~ s/\s+//g;
  $b{name} =~ s/\s+//g;

  ($a{name}, $a{fct}) = ($2, $1) if $a{name} =~ /(exp|inv\.logit)\(([^\)]*)\)/;
  ($b{name}, $b{fct}) = ($2, $1) if $b{name} =~ /(exp|inv\.logit)\(([^\)]*)\)/;

  ($a{name}, $a{dim}) = ($`, $1) if $a{name} =~ /\[(.*)\]/;
  ($b{name}, $b{dim}) = ($`, $1) if $b{name} =~ /\[(.*)\]/;


  if ($a{name} eq $b{name})
  {
    $a{fct} cmp $b{fct} || &DimsOrder($a{dim}, $b{dim});
  }
  else
  {
    &StringsOrder($a{name}, $b{name});
  }
} # end of SortNodes


sub StatsReport()
{
  my (@stats) = @_;
  my @report;


  foreach my $stat (@stats)
  {
    if ($stat eq "mcError")
    {
      push(@report, $stat) if $IncludeMCStats;
    }
    elsif ($stat eq "start" || $stat eq "sample")
    {
      push(@report, $stat) if $IncludeMCStats && !$ConstantNIter;
    }
    else
    {
      push(@report, $stat);
      
      push(@report, "crIImbalance") if $stat eq "ucl" && $showCrIImbalance;
    }
  }

  @report;
} # end of StatsReport


sub StringsOrder()
{
  my ($a, $b) = @_;
  my ($res, %a, %b);

  # (where a string can be a series of characters and/or digits)

  ($a{len}, $b{len}) = (length($a), length($b));


  if ($a{len} == 0)
  {
    $res = $b{len} == 0 ? 0 : -1;
  }
  elsif ($b{len} == 0)
  {
    $res = 1;
  }
  else
  {
    # both strings are non-null

    ($a{left}, $a{num}, $a{right}) = ($`, $&, $') if $a =~ /[\+\-\.]?\d+/;
    ($b{left}, $b{num}, $b{right}) = ($`, $&, $') if $b =~ /[\+\-\.]?\d+/;

    $a{numlen} = length($a{num});
    $b{numlen} = length($b{num});


    if ($a{numlen})
    {
      if ($b{numlen})
      {
        if ($a{left})
        {
          if ($b{left})
          {
            $res = &WordsOrder($a{left}, $b{left}) || &StringsOrder($a{num} . $a{right}, $b{num} . $b{right});
          }
          else
          {
            $res = 1;
          }
        }
        elsif ($b{left})
        {
          $res = -1;
        }
        else
        {
          # both a and b start with numbers

          if ($a{right} =~ /\.\d+/ && !$`)
          {
            unless ($a{num} =~ /\./)
            {
              $a{right} =~ /\.\d+/;
              $a{num} .= $&;
              $a{right} = $';
            }
          }

          if ($b{right} =~ /\.\d+/ && !$`)
          {
            unless ($b{num} =~ /\./)
            {
              $b{right} =~ /\.\d+/;
              $b{num} .= $&;
              $b{right} = $';
            }
          }

          $res = $a{num} <=> $b{num} || &StringsOrder($a{right}, $b{right});
        }
      }
      else
      {
        if ($a{left})
        {
          $res = &WordsOrder($a{left}, $b) || 1;
        }
        else
        {
          $res = -1;
        }
      }
    }
    elsif ($b{numlen})
    {
      if ($b{left})
      {
        $res = &WordsOrder($a, $b{left}) || -1;
      }
      else
      {
        $res = 1;
      }
    }
    else
    {
      $res = &WordsOrder($a, $b);
    }
  }

  $res;
} # end of StringsOrder


sub StrNumStr()
{
  my ($str) = @_;
  my (%left, %num, %right);
  
  if ($str =~ /\d+\.\d*/)
  {
    ($left{1}, $num{1}, $right{1}) = ($`, $&, $');
  }
  
  if ($str =~ /\d*\.\d+/)
  {
    ($left{2}, $num{2}, $right{2}) = ($`, $&, $');
  }
  
  if ($str =~ /\d+/)
  {
    ($left{3}, $num{3}, $right{3}) = ($`, $&, $');
  }
  
  if ($str =~ /[\+\-]\d+\.?\d*/)
  {
    ($left{4}, $num{4}, $right{4}) = ($`, $&, $');
  }
  
  my ($k) = sort {length($left{$a}) <=> length($left{$b})} keys %left;
  my @out = ($left{$k}, $num{$k}, $right{$k}) if $k;
  
  @out;
} # end of StrNumStr


sub TitleUsedInMostRecentWinBUGSSummaryFileSameDir()
{
  my (@dirs) = @_;
  my ($title, @html);

  # Global variables:
  # -----------------
  # %monitoredLast

  foreach my $dir (@dirs)
  {
    next unless -d $dir;
    push(@html, <$dir/*.html>);
  }


  if (@html)
  {
    my %age;

    foreach (@html)
    {
      $age{$_} = -M;
    }

    my ($html) = sort {$age{$a} <=> $age{$b}} @html;

    open(HTML, $html);
    while (<HTML>)
    {
      if (/\<div class=title\>(.*)\<\/div\>/i)
      {
        $title = $1;
        $title =~ s/\s*(.*)/\1/g; # ltrim
        $title =~ s/\s+$//;       # rtrim
      }
      elsif (/\<td\s+[^\>]*\sclass\s*=\s*header[^\>]*\>([^\<]*)\<\/td\>/i)
      {
        my $tmp = $1;
        $tmp =~ s/\s*(.*)/\1/g; # ltrim
        $tmp =~ s/\s+$//;       # rtrim
        $tmp =~ s/\[.*//;
        $monitoredLast{$tmp} = 1;
      }
    }
    close HTML;
  }

  $title;
} # end of TitleUsedInMostRecentWinBUGSSummaryFileSameDir


sub Unique()
{
  my (@elem) = @_;
  my %z;
  
  foreach my $elem (@elem)
  {
    $z{$elem} = 1;
  }

  keys %z;
} # end of Unique


sub WordsOrder()
{
  my ($a, $b) = @_;

  # Words do not include any digit

  if ($a eq $b)
  {
    0;
  }
  else
  {
    my ($lca, $lcb) = (lc $a, lc $b);

    $lca =~ s/[^a-z]//g;
    $lcb =~ s/[^a-z]//g;

    $lca cmp $lcb || lc $a cmp lc $b || $a cmp $b;
  }
} # end of WordsOrder
