# >>> WriteWinBUGSScript version 2.1 <<<
#                  (Released November 2015)
#
# This program was written by
# ---------------------------
# Patrick Belisle
# Division of Clinical Epidemiology
# McGill University Health Centre
# Montreal, Qc CANADA
# patrick.belisle@clinepi.mcgill.ca

# For instructions on how to use this pgm, please visit
# http://www.medicine.mcgill.ca/epidemiology/Joseph/PBelisle/WriteWinBUGSScript.html

# Nouveautes version 1.11 (a documenter)
# ------------------------------------------------------------------------------
# -i some.node=dbern(p=0.5)  <for 0/1 data node with NA's to impute on>
# -i some.node=dbern(logit.p=-0.23)  <for 0/1 data node with NA's to impute on>
# -i chance.node=dbern(0.6){C}
# -i chance.node=dbern(0.7){15}


# ----- Do not edit below this line -----------------------------------------------------------------------------------------


use strict;
use Win32;



our ($LogFile, $MyStartTime);
&OpenSTDERR($0, 1);


srand(time()^($$+($$<<16))); # rnd seed (as in book "Advanced Perl"), in case we need it (needed in sub RndValues)



die "No arguments given. Abort.\n" unless @ARGV;

my @Arguments = &SrcScripts(@ARGV);

# ------------------------------------------------------------------------------------------

our @RndZ = (0.622, -0.412, -1.952,  0.927, -1.176, -0.480,  0.189, -0.683, -0.309, -0.411,
            -0.227,  0.007, -1.274,  0.578,  1.099,  0.819, -0.099,  3.373, -0.047, -0.183,
            -0.279,  0.584, -0.665, -1.438,  0.888, -0.690, -1.528, -2.058, -2.737, -0.351,
            -0.950,  0.336, -0.348,  0.123,  1.558,  1.004, -1.329, -1.757, -1.270,  0.598,
             0.495, -0.089, -0.074,  0.249,  0.380, -0.425,  1.067, -1.550, -0.582, -0.963,
            -0.639, -0.081, -0.366,  0.109,  0.473, -1.502,  0.257,  1.454, -0.362,  1.990,
             0.115, -0.615, -0.452, -0.772, -0.431, -1.073,  0.061, -1.098,  2.141, -1.569,
             0.362,  1.755,  0.471, -0.700,  0.511, -0.622,  1.009, -2.279, -0.194,  0.333,
            -0.584, -0.257, -0.056, -0.620,  1.587, -1.077, -0.205,  1.521,  0.706,  2.045,
             0.431, -0.475, -3.132, -0.982,  0.583,  1.544, -2.099, -1.188,  0.461, -0.453);

# ------------------------------------------------------------------------------------------
our ($RmScientificNotationInWinBUGSDatafileNModif, $scriptno);

my @Tmpdir;
our @WrittenScripts;

our %action;
our %Dir;
our %ParmValue;
our %VarIndexed;

# ------------------------------------------------------------------------------------------

# $Dir{tmp} = "c:/my computer/my tmp dir";

# If you want to specify in which directory temporary inits files are to be saved, please do so by uncommenting the line above and
# quoting the actual path to your tmp files directory (make sure to use slashes (/), not backslashes (\), in its path, though).


my $CWD = Win32::GetCwd();




$Dir{tmp}    = Win32::GetShortPathName($ENV{TMP} || $ENV{TEMP});
$Dir{tmp}    =~ s{\\}{/}g;
$Dir{wb}     = "$Dir{tmp}/WinBUGS";
$Dir{inits}  = "$Dir{wb}/inits";
$Dir{parms}  = "$Dir{wb}/parms";


foreach ("inits", "parms")
{
  &MkDir($Dir{$_}) unless -d $Dir{$_};
  $Dir{$_} = Win32::GetFullPathName($Dir{$_});
  $Dir{$_} =~ s{\\}{/}g;
}


my $date8 = &Date8();

our %Tmp = (inits => $Dir{inits} . "/" . $date8,
            parms => $Dir{parms} . "/" . $date8);

@Tmpdir = ($Dir{wb}, $Dir{parms}, $Dir{inits});

our %Width = (inits => 120, parms => 120);

our %dbernKey = (p    => "p",
                 prob => "p",
                 logitp       => "logitp",
                 logitprob    => "logitp",
                 "logit.p"    => "logitp",
                 "logit.prob" => "logitp");

# ------------------------------------------------------------------------------
chdir($CWD);

my $tmp = shift @Arguments;
my $lctmp = lc $tmp;

if ($lctmp eq "-cl")
{
  # -cl must be used as a first argument when WriteWinBUGSScript is called from DOS-prompt

  $_ = join(" ", @Arguments);
  s/'/"/g;
  my @instructions = &OneLineInstructions($_);
  &WriteScript(@instructions);
}
elsif ($lctmp eq "-clean" || $lctmp eq "-rm")
{
  my $nfilesRemoved;

  foreach (@Tmpdir)
  {
    chdir($_) || die "Cannot cd to tmp dir $_\nSorry\n";
    my @unlink = (<*.txt>);
    $nfilesRemoved += @unlink;
    unlink(@unlink);
  }

  if ($nfilesRemoved)
  {
    my $s = $nfilesRemoved > 1 ? "s" : "";
    print STDERR "Removed $nfilesRemoved file$s.\n";
  }
  else
  {
    print STDERR "No tmp file to remove.\n";
  }

  print STDERR "Done\n";
}
else
{
  my $skip;
  unshift(@Arguments, $tmp);

  foreach my $arg (@Arguments)
  {
    my @instructions;

    open(MODELS, $arg) || die "Cannot read WriteWinBUGSScript arguments from file $arg. Abort.\n";

    ($CWD) = Win32::GetFullPathName($arg);
    chdir $CWD;

    while (<MODELS>)
    {
      chomp;
      s/\t/ /g;
      s/'/"/g;

      if (/\S/)
      {
        $skip = 1 if /#\s*skip/i;

        s/#.*//;
        next unless /\S/;

        if (/\-(\S+)/)
        {
          my $untouched = $_;

          my ($option, $after) = (lc $1, $');
          $_ = "-" . $option . $after;

          if ($option eq "1l")
          {
            @instructions = &OneLineInstructions($after);
            &WriteScript(@instructions) unless $skip;
            undef $skip;
            undef @instructions;
            next;
          }
          else
          {
            push(@instructions, $untouched);
          }
        }
        else
        {
          unshift(@instructions, $_);
        }
      }
      elsif (@instructions)
      {
        &WriteScript(@instructions) unless $skip;
        undef @instructions;
        undef $skip;
      }
    }
    close MODELS;

    &WriteScript(@instructions) if @instructions && !$skip;
  }
}



if (@WrittenScripts)
{
  my $s = $#WrittenScripts > 0 ? "s" : "";

  print STDERR "\nWritten script$s:\n";

  foreach (@WrittenScripts)
  {
    print STDERR "\t" . $_ . "\n";
  }
}


print STDERR "\nDone.\n";


# ---- End of code -------------------------------------------------------------


sub byDim()
{
  my ($a, $b) = @_;
  my (@a, @b, $res);

  @a = split(",", $a);
  @b = split(",", $b);

  while (!$res && (@a || @b))
  {
    ($a, $b) = (shift @a, shift @b);
    $res = $a <=> $b;
  }

  $res;
} # end of byDim


sub byVarNamesCaseInsensitive()
{
  my ($res, $x, $y, $xdim, $ydim, @tmp, @num);
  $res = 0;

  # fct used to sort SAS/WinBUGS variable names (or files by filenames) taking into account the numeric values of the numbers in the names
  # (e.g. x2 must come before x12, which would not be the case would we sort in alphabetical order)
  # Note: non-digits and non-characters are ignored in sorting args

  # Note that (numeric) dimensions are used in case of equalities

  ($x, $y) = (lc $a, lc $b);

  if ($x =~ /\[(.*)\]/)
  {
    ($x, $xdim) = ($`, $1);
  }


  if ($y =~ /\[(.*)\]/)
  {
    ($y, $ydim) = ($`, $1);
  }


  foreach ($x, $y)
  {
    s/[^A-Za-z0-9]//g;
  }

  while (!$res && length($x) > 0)
  {
    @tmp = ();
    @num = ();

    foreach ($x, $y)
    {
      if (/\d+/)
      {
        push(@tmp, $`);
        push(@num, $&);
        $_ = $';
      }
      else
      {
        push(@tmp, $_);
        push(@num, "");
        $_ = "";
      }
    }

    $res = ($tmp[0] cmp $tmp[1]) || ($num[0] <=> $num[1]);    
  }


  if (!$res && (length($xdim) || length($ydim)))
  {
    &byDim($xdim, $ydim);
  }
  else
  {
    $res;
  }
} # end of byVarNamesCaseInsensitive


sub Clean()
{
  my ($txt) = @_;
  my ($new, $tmp);

  $txt =~ s/\s*=\s*/=/g;

  while ($txt =~ /\s*\{[^\}]*\}/)
  {
    ($tmp, $txt) = ($&, $');
    $new .= $`;
    $tmp =~ s/\s+//g;
    $new .= $tmp;
  }

  $new .= $txt;
  $txt = $new;
  undef $new;
  
  while ($txt =~ /\s*\([^\)]*\)/)
  {
    ($tmp, $txt) = ($&, $');
    $new .= $`;
    $tmp =~ s/\s+//g;
    $new .= $tmp;
  }

  $new . $txt;
} # end of Clean


sub Date8()
{
  my ($mday, $mon, $year, @tmp);
  
  @tmp = localtime (time);
  ($mday, $mon, $year) = splice(@tmp, 3, 3);
  $year += 1900;
  $mon++;
  $mon = "0$mon" if $mon < 10;
  $mday = "0$mday" if $mday < 10;

  "$year$mon$mday";
}


sub dbern()
{
  my ($p) = @_;
  my $u = rand; # Unif(0,1)
  $u < $p ? 1 : 0;
} # end of dbern


sub ExtractFilename()
{
  my ($tmp) = @_;
  my @tmp;

  $tmp =~ s/'/"/g;
  $tmp =~ s/\s*(.*)/\1/g; # ltrim

  if (substr($tmp, 0, 1) eq "\"")
  {
    $tmp =~ /"([^"]*)"\s*/;
    @tmp = ($1, $');
  }
  else
  {
    $tmp =~ /(\S+)\s*/;
    @tmp = ($1, $');
  }

  @tmp;
} # end of ExtractFilename


sub FirstAvailableFilenameWithNumerixExtension()
{
  my ($path, $ext, $extlen) = @_;
  my $k;
  
  $path .= "-";
  $ext = "." . $ext if $ext && substr($ext, 0, 1) ne ".";
  
  my $newfile = $path . &StrNo(++$k, $extlen) . $ext;
  $newfile = $path . &StrNo(++$k, $extlen) . $ext while -e $newfile;
  
  $newfile;
} # end of FirstAvailableFilenameWithNumerixExtension


sub InvLogit()
{
  my (@x) = @_;

  foreach (@x)
  {
    $_ = exp($_);
    $_ /= ($_ + 1);
  }

  # Return as vector or scalar (as it was received)

  if ($#x)
  {
    @x;
  }
  else
  {
    $x[0];
  }
} # end of InvLogit


sub IsInteger()
{
  my ($x) = @_;
  my ($y, $res);
  
  # Attention: 
  # $x = 12.0   retournera "oui": donc reconnu comme valeur entire
  # $x = "12.0" retournera "non"
  # $x = 1e2    retournera "oui"
  # $x = "1e2"  retournera "non"
  # $x = 12     retournera "oui"
  # $x = "12"   retournera "oui"
  #
  # NOTE: si on utilise &IsInteger($x+0), on retourne "oui" pour tous les ex. ci-haut
  #
  
  $x =~ s/\s+//g;
  $y = sprintf("%d", $x);
  $res = (($x eq $y) ? 1 : 0);
  $res;
} # end of IsInteger


sub IsListFile()
{
  my ($file) = @_;
  my $isListFile;
  
  open(TMP, $file);
  while ($tmp = <TMP>)
  {
    my $tmp =~ s/#.*//;
    next unless $tmp =~ /\S/;
    $isListFile = 1 if $tmp =~ /\s*list\(/ && !$`;
    last;
  }
  close TMP;
    
  $isListFile || 0;
} # end of IsListFile


sub IsScript()
{
  my ($file) = @_;
  my $isscript = 0;
  
  open(TMP, $file);
  while(<TMP>)
  {
    if (/\s*check\(.*\)/ && !$`)
    {
      $isscript = 1;
      last;
    }
  }
  close(TMP);
  
  $isscript;
} # end of IsScript


sub ListValues()
{
  # read c() values as well as structure()'s

  my ($file, @objNames) = @_;
  my ($nBrackets, $object, $readingValues,
      %isObject, %values);

  open(WBLIST, $file);
  my @tmp = (<WBLIST>);
  close WBLIST;
  chomp @tmp;

  foreach (@objNames)
  {
    $isObject{$_} = 1;
  }

  my $tmp = shift @tmp;

  while ()
  {
    if ($readingValues)
    {
      if ($tmp =~ /[\(\)]/)
      {
        my $match = $&;
        $tmp = $';
        $values{$object} .= $` . $match;

        $nBrackets += $match eq "(" ? 1 : -1;

        unless ($nBrackets)
        {
          undef $readingValues;
          delete $isObject{$object};
          undef $object;

          last unless keys %isObject;
        }

        redo if $tmp =~ /\S/;
      }
      else
      {
        $values{$object} .= $tmp;
      }
    }
    elsif ($tmp =~ /[\(\s,](\S+)\s*=\s*(c|structure)\(/ || ($tmp =~ /(\S+)\s*=\s*(c|structure)\(/ && !$`))
    {
      my $obj = $1;
      $tmp = $';

      if ($isObject{$obj})
      {
        ($object, $readingValues, $nBrackets) = ($obj, 1, 1);
        $values{$object} = $2 . "(";
      }

      redo if $tmp =~ /\S/;
    }


    last unless @tmp;
    $tmp = shift @tmp;
  }

  # remove consecutive blanks
  foreach (values %values)
  {
    s/\s+/ /g;
  }

  %values;
} # end of ListValues


sub ListVarnamesAndOrConstants()
{
  my ($file, $nodie, $returnallvarnames, $returnscalarvalues) = @_;

  my ($data, $dlm, $value, $var,
      @res);


  # NOTE: this fct is not adapted for lists with character string variables (which won't happen in WinBUGS anyhow!)

  open(DATA, $file);
  while (<DATA>)
  {
    s/#.*//;
    next unless /\S/;
    chomp;
    $data .= " " . $_;
  }
  close DATA;


  while ($data =~ /=/)
  {
    ($var, $data) = ($`, $');

    $var =~ s/\s+$//; # rtrim
    $var =~ s/.*[,\(\s]//;
    $var = $1 if $var =~ /.*\s(\S+)/;

    if ($var eq ".Data" || $var eq ".Dim")
    {
      $data =~ s/[^\)]*\)//;
    }
    else
    {
      $data =~ /[,\(\)]/;
      ($value, $dlm, $data) = ($`, $&, $');

      $value =~ s/\s*(.*)/\1/g; # ltrim

      if ($value eq "c" && $dlm eq "(")
      {
        push(@res, $var) if $returnallvarnames;
        $data =~ s/[^\)]*\)//;
      }
      else
      {
        $value =~ s/\s+$//; # rtrim
        push(@res, "$var=$value");
      }
    }
  }

  @res;
} # end of ListVarnamesAndOrConstants


sub MkDir()
{
  my (@dirs) = @_;
  my ($cwd, $dir, $notFound, $subdir,
      @notFound);

  $cwd = Win32::GetCwd();

  foreach $dir (@dirs)
  {
    next if -d $dir;
    $dir =~ s{/}{\\}g;
    undef @notFound;

    while ($dir =~ /(.*)\\/)
    {
      ($dir, $notFound) = ($1, $');
      unshift(@notFound, $notFound);
      last if -d $dir;
      last unless $dir =~ /\\[^\\]+\\/;
    }

    if (-d $dir)
    {
      chdir($dir);
      foreach $subdir (@notFound)
      {
        mkdir($subdir, 0777);
        chdir($subdir);
      }
    }
  }

  chdir($cwd);
} # end of MkDir


sub MyChr()
{
  my ($i) = @_;
  my $j;

  # Turns 1 into a, 2 into b, ..., 26 into z, 27 into aa, 28 into ab, ..., 52 into az, etc.
  # useful for file names suffixes

  if ($i <= 26)
  {
    chr(96 + $i);
  }
  else
  {
    $j = int (($i-1)/26);
    &MyChr($j) . chr(97 + ($i-1)%26);
  }
} # end of MyChr


sub NodeValues()
{
  my (@tmp) = @_;
  my ($tmp, @out);
  
  foreach (@tmp)
  {
    s/#.*//; # remove commented out values
  }
  
  while (@tmp)
  {
    my ($matchtype, $node, $value);
    
    unless ($tmp =~ /[=~]/)
    {
      while (@tmp)
      {
        $tmp .= shift @tmp;
        last if $tmp =~ /[=~]/;
      }
    }
  
    last unless $tmp =~ /[=~]/;

    ($node, $matchtype, $tmp) = ($`, $&, $');
    $node =~ s/^\s+|\s+$//g; # lrtrim
    

    until ($tmp =~ /[,\(\)]/)
    {
      $tmp .= shift @tmp;
      last unless @tmp;
    }
    
    last unless $tmp =~ /[,\(\)]/;
    my ($left, $match, $right) = ($`, $&, $');
    
    if ($match eq "," || $match eq ")")
    {
      $value = $left;
      $value =~ s/^\s+|\s+$//g; # lrtrim
      
      $tmp = $right;
      $tmp .= shift @tmp until $tmp =~ /\S/ || !@tmp;
      
      if ($match eq ")")
      {
        $tmp .= shift @tmp until $tmp =~ /\S/ || !@tmp;
        $tmp =~ s/,\s*//;
      }
    }
    else
    {
      ($value, $tmp) = ($left . $match, $right);
      my $nBrackets = 1;
      
      while ($nBrackets)
      {
        my ($bracket, $left);
        
        until ($tmp =~ /[\(\)]/)
        {
          last unless @tmp;
          $tmp .= shift @tmp;
        }

        last unless $tmp =~ /[\(\)]/;
        
        ($left, $bracket, $tmp) = ($`, $&, $');
        $value .= $left . $bracket;
        $nBrackets += $bracket eq "(" ? 1 : -1;
      }
      
      last if $nBrackets;
      
      $value =~ s/^\s+|\s+$//g; # lrtrim
      
      # remove leading comma
      $tmp .= shift @tmp until $tmp =~ /\S/ || !@tmp;
      $tmp =~ s/,\s*//;
    }
    
    push(@out, $node, $matchtype, $value);
  }
  
  
  @out;
} # end of NodeValues


sub NoTauWhenSD()
{
  my (@vars) = @_;
  my (%exists, %varno, $k, $var, $sdvar, $sigmavar, $varbody, $vartail);

  # To eliminate variables which names end by '.tau|.prec' from a list
  # when the corresponding '.sd|.sigma' variable exists
  # (e.g., u.tau is deleted iff u.sd is part of @vars)
  #
  # Note: variables will be returned in same order as received.

  foreach (@vars)
  {
    $varno{$_} = $k++;
    $exists{$_} = 1;
  }

  foreach $var (@vars)
  {
    if ($var =~ /(.*)\./)
    {
      ($varbody, $vartail) = ($1, $');
      next unless $vartail eq "tau" || $vartail eq "prec";

      $sdvar    = $varbody . ".sd";
      $sigmavar = $varbody . ".sigma";
      delete $exists{$var} if $exists{$sdvar} || $exists{$sigmavar};
    }
  }

  sort {$varno{$a} <=> $varno{$b}} keys %exists;
} # end of NoTauWhenSD


sub NotScientific()
{
  my ($x) = @_;
  my ($e, $negative, $lx);

  if ($x =~ /-?\d+\.\d+e\-\d+/i || $x =~ /-?\d+e\-\d+/i)
  {
    $x =~ /e/i;
    ($x, $e) = ($`, $');
    $x =~ s/\.//;
    ($negative, $x) = (1, substr($x,1)) if substr($x,0,1) eq "-";
    
    $x = "0." . "0" x (-$e-1) . $x;
  }
  elsif ($x =~ /-?\d+\.\d+e\+?\d+/i || $x =~ /-?\d+e\+?\d+/i)
  {
    $x =~ /e/i;
    ($x, $e) = ($`, $');
    $x =~ s/\.//;
    ($negative, $x) = (1, substr($x,1)) if substr($x,0,1) eq "-";
    $e++;
    
    $lx = length($x);
    if ($e < $lx)
    {
      substr($x, $e, 0) = ".";
    }
    elsif ($e > $lx)
    {
      $x .= "0" x ($e-$lx);
    }
  }

  $x = "-" . $x if $negative;
  $x;
} # end of NotScientific


sub OneLineInstructions()
{
  my ($instr) = @_;
  my ($current, $new, @instr);

  $instr =~ s/'/"/g;

  # remove spaces between '=' and dashes (as in b.age = -2)

  while ($instr =~ /"([^"]*)"/)
  {
    my ($left, $tmp);
    ($left, $tmp, $instr) = ($`, $&, $');
    $new .= &Clean($left) . $tmp;
  }
  $new .= &Clean($instr);

  $instr = $new;
  $instr =~ s/\s+$//; # rtrim

  # split 1-liner instructions in multiple-lines, with flags put in 1st column

  while ($instr =~ /\S/)
  {
    $instr =~ s/\s*(.*)/\1/g; # ltrim

    if (substr($instr, 0, 1) eq "\"")
    {
      $instr =~ /"([^"]*)"/;
      $current .= " " . $&;
      $instr = $';
    }
    elsif (substr($instr, 0, 1) eq "-")
    {
      push(@instr, $current) if $current;
      /(\S+)\s*/;
      ($current, $instr) = ($1, $');
    }
    else
    {
      /(\S+)\s*/;
      $current .= " " . $1;
      $instr = $';
    }
  }

  push(@instr, $current) if $current;

  foreach (@instr)
  {
    s/\s*(.*)/\1/g; # ltrim
  }

  @instr;
} # end of OneLineInstructions


sub OpenSTDERR()
{
  my ($callingpgm, $timestamp, $definegloballogfile) = @_;
  my ($logfile, $logdir);

  # Global variables defined:
  # ------------------------------------------------------- 
  # $LogFile (optionally, that is, if $definegloballogfile)
  # $MyStartTime


  $logfile = "$1.log" if $callingpgm =~ /(.*)\./;
  ($logdir, $logfile) = Win32::GetFullPathName($logfile);
  $logdir = $& . "log\\" if $logdir =~ /c\:\\users\\patrick\.belisle\\My Documents\\Home\\bin\\/i || $logdir =~ /c\:\\users\\patrick\.belisle\\Documents\\Home\\bin\\/i;
  $logfile = $logdir . $logfile;

  open(STDERR, ">$logfile") || die "Cannot write to log file $logfile. Abort.\n";
  print STDERR localtime() . "\n\n" if $timestamp;

  $MyStartTime = time; # defined for future use, as in fct ReportRunTime, for example

  $LogFile = $logfile if $definegloballogfile;
}


sub ParmsFileName()
{
  my ($parms) = @_;
  my $fname;
  my $writeNew = 1;

  # List all parms files already found in Temp

  chdir($Dir{parms});
  my @files = (<*.txt>);

  # Read each parms file aready found in Temp,
  # and see if any corresponds to the parms that need to be saved now

  foreach my $file (@files)
  {
    open(TMP, $file);
    my @tmp = (<TMP>);
    close TMP;

    chomp @tmp;
    while (@tmp)
    {
      my $tmp = shift @tmp;
      if ($tmp =~ /\s*list\(/ && !$`)
      {
        $tmp = $`;
        unshift(@tmp, $tmp) if $tmp =~ /\S/;
        last;
      }
    }

    next unless @tmp;

    while (@tmp)
    {
      my $tmp = pop @tmp;
      if ($tmp =~ /\)/)
      {
        $tmp = $`;
        push(@tmp, $tmp) if $tmp =~ /\S/;
        last;
      }
    }

    $tmp = join(" ", @tmp); # joined parms found in file read
    $tmp =~ s/\s*,\s*/, /g;
    $tmp =~ s/^\s+|\s+$//g; # lrtrim

    if ($tmp eq $parms)
    {
      ($writeNew, $fname) = (0, $file);
      last;
    }
  }


  $fname = &FirstAvailableFilenameWithNumerixExtension($Tmp{parms}, "txt", 2) if $writeNew;
  $fname = Win32::GetFullPathName($fname);

  ($writeNew, $fname);
} # end of ParmsFileName


sub ParmValue()
{
  my ($parm, $file) = @_;
  my $tmp;
  
  # Global hash table: %ParmValue
  
  open(TMP, $file) || die "Cannot read file $file (looking for value of parm $parm)\n";
  while (<TMP>)
  {
    s/#.*//;
    $tmp .= $_ if /\S/;
  }
  close TMP;

  $tmp =~ s/[\(\)]/,/g;
  $tmp =~ s/\s+//g;
  
  if ($tmp =~ /,$parm=([^,]+),/)
  {
    $ParmValue{$parm}{$file} = $1;
    $ParmValue{$parm}{$file};
  }
  else
  {
    die "Parameter $parm not defined in file $file\nSorry\n";
  }
} # end of ParmValue


sub Rep()
{
  my ($value, $nrep, $firstna) = @_;
  my $res = "c(";

  if ($firstna && $nrep)
  {
    $nrep--;
    $res .= "NA,"
  }

  $res .= "$value," x $nrep;
  chop $res; # remove trailing comma

  $res . ")";
} #end of Rep


sub RmScientificNotationInWinBUGSDatafile()
{
  my ($file) = @_;
  my ($anyModif, $atime, $comment, $left, $mtime, $tmp,
      @mtime, @new, @tmp);

  # Global variable:
  # ----------------
  # $RmScientificNotationInWinBUGSDatafileNModif
      
  # This function will work correctly only if c:\users\patrick.belisle\My Documents\Home\bin\PerlFcts\math.txt is require'd
  
  open(TMP, $file);
  @tmp = (<TMP>);
  close TMP;
  chomp @tmp;
  
  while (@tmp)
  {
    $tmp = shift @tmp;
    
    undef $comment;
    undef $left;
    
    ($tmp, $comment) = ($`, $&) if $tmp =~ /#.*/;
    
    while ($tmp =~ /(\-?\d\.\d+e\-?\d+)\s*(\D)/i)
    {
      if ($2 eq "=")
      {
        $left .= $` . $&;
        $tmp = $';
      }
      else
      {
        $tmp = $';
        $left .= $` . &NotScientific($1) . $2;
        #$left .= $` . "***>" . $1 . "<***" . $2;
        $anyModif = 1;
      }
    }
    
    if ($left)
    {
      $tmp = $left . $tmp;
      undef $left;
    }
    
    while ($tmp =~ /(\-?\d+e\-?\d+)\s*(\D)/i)
    {
      if ($2 eq "=")
      {
        $left .= $` . $&;
        $tmp = $';
      }
      else
      {
        $tmp = $';
        $left .= $` . &NotScientific($1) . $2;
        $anyModif = 1;
      }
    }  
    
    push(@new, $left . $tmp . $comment);
  }
  
  if ($anyModif)
  {
    # Note file date and time, leave it unchanged after file is modified
    @mtime = stat($file);
    ($atime, $mtime) = splice(@mtime, 8, 2);
    
    open(TMP, ">$file");
    foreach (@new)
    {
      print TMP $_ . "\n";
    }
    close TMP;
    
    # Put back original file date and time
    utime($atime, $mtime, $file);

    $RmScientificNotationInWinBUGSDatafileNModif++;
  }
} # end of RmScientificNotationInWinBUGSDatafile


sub RndIntegers()
{
  my ($n, $replacement, $max, $min) = @_;
  my (@rnd, 
      $x, $mvalues,
      %sampled);

  # Any pgm using this macro should do line below once at the beginning of the pgm:
  # srand(time()^($$+($$<<16))); # set seed (as suggested in Advanced Perl manual)

  # Sample $n integer values uniformly from $min..$max (Default for $min = 1)
  # with or without replacement, according to value in $replacement (0/1)

  $min = 1 unless defined $min;
  $mvalues = $max - $min + 1;

  die "Cannot sample $n values from $min to $max without replacement.\n" if !$replacement && $n > $mvalues;


  foreach (1..$n)
  {
    $x = sprintf("%d", rand $mvalues) + $min;

    unless ($replacement)
    {
      redo if $sampled{$x};
      $sampled{$x} = 1;
    }
   
    push(@rnd, $x);
  }

  @rnd;
} # end of RndIntegers


sub RndValues()
{
  my ($values, $m) = @_;
  my ($p, $pcum, $res);

  ($values, $p) = ($`, $') if $values =~ /\:/;
  my @values = split(",", $values);
  my @p = split(",", $p);

  # Standardize and cumulate probabilities vector

  foreach (@p)
  {
    $pcum += $_;
    $_ = $pcum;
  }
  foreach (@p)
  {
    $_ /= $pcum;
  }


  # sample $m values

  foreach (1..$m)
  {
    $p = rand;
    my $i = 0;
    $i++ while $p > $p[$i];

    $res .= ", " . $values[$i];
  }

  $res =~ s/,\s*//; # remove leading comma (and following blanks)

  "c(" . $res . ")";
} # end of RndValues


sub ShortLines()
{
  my ($width, @lines) = @_;
  my (@res, $current, $left, $left0, $line, $margin, $quote, $tmp);

  chomp @lines;

  foreach $line (@lines)
  {
    if (length($line) <= $width)
    {
      push(@res, $line);
      next;
    }
    else
    {
      $line =~ /\S/;
      $margin = " " x (length($`) + 2);
    }

    undef $current;

    while ($line =~ /\S/)
    {
      if ($line =~ /['"]/)
      {
        $quote = $&;

        if ($quote eq "'")
        {
          if ($line =~ /'[^']*'[^,]*,/)
          {
            ($left, $tmp, $line) = ($`, $&, $');

            while ($left =~ /,/)
            {
              ($left0, $left) = ($` . $&, $');
              $current .= $left0;
              if (length($current) >= $width)
              {
                push(@res, $current);
                $current = $margin;
              }
            }

            $current .= $left . $tmp;
            if (length($current) >= $width)
            {
              push(@res, $current);
              $current = $margin;
            }
          }
          else
          {
            push(@res, $current . $line);
            undef $line;
            $current = $margin;
          }
        }
        else
        {
          # $quote eq "\""
          
          if ($line =~ /"[^"]*"[^,]*,/)
          {
            ($left, $tmp, $line) = ($`, $&, $');

            while ($left =~ /,/)
            {
              ($left0, $left) = ($` . $&, $');
              $current .= $left0;
              if (length($current) >= $width)
              {
                push(@res, $current);
                $current = $margin;
              }
            }

            $current .= $left . $tmp;
            if (length($current) >= $width)
            {
              push(@res, $current);
              $current = $margin;
            }
          }
          else
          {
            push(@res, $current . $line);
            undef $line;
            $current = $margin;
          }
        }
      }
      elsif ($line =~ /,/)
      {
        $line = $';
        $current .= $` . $&;

        if (length($current) >= $width)
        {
          push(@res, $current);
          $current = $margin;
        }
      }
      else
      {
        $current .= $line;
        push(@res, $current);
        undef $line;
        undef $current;
      }
    }

    $current .= $line;
    push(@res, $current) if $current =~ /\S/;
  }

  @res;
}


sub SlashFullPathName()
{
  my ($file) = @_;
  my ($wb);

  $wb = Win32::GetFullPathName($file);
  $wb =~ s{[\\/]+}{/}g;

  $wb;
} # end of SlashFullPathName


sub SrcScripts()
{
  my (@files) = @_;
  my %script2rewrite;
  my $allscripts = 1;

  foreach my $file (@files)
  {
    unless (&IsScript($file))
    {
      $allscripts = 0;
      last;
    }
  }


  if ($allscripts)
  {
    my %file2src;

    foreach my $file (@files)
    {
      $file = Win32::GetFullPathName($file);
      $script2rewrite{$file} = 1;
    }

    my ($dir) = Win32::GetFullPathName($files[0]);
    chdir($dir);
    chdir("src");
    my @tmp = (<*.txt>);

    foreach my $file (@tmp)
    {
      my $file2src;

      open(TMP, $file);
      while (<TMP>)
      {
        if (/\s*\-s\s+/ && !$`)
        {
          my $tmp = $';
          $tmp =~ s/\s+$//; # rtrim
          next unless -f $tmp;
          $tmp = Win32::GetFullPathName($tmp);

          if ($script2rewrite{$tmp})
          {
            $file2src = 1;
            last;
          }
        }
      }
      close TMP;

      next unless $file2src;
      $file = Win32::GetFullPathName($file);
      $file2src{$file} = 1;
    }

    keys %file2src;
  }
  else
  {
    @files;
  }
} # end of SrcScripts


sub Star()
{
  my ($action, $var) = @_;
  my $res;

  foreach my $codavar (keys %{$action{$action}})
  {
    if ($var =~ /$codavar/ && !$` && !$')
    {
      $res = 1;
      last;
    }
  }

  $res;
} # end of Star


sub StrNo()
{
  my ($no, $len) = @_;
  my $l = length($no);
  
  if ($l < $len)
  {
    my $tmp = "0" x ($len - $l);
    $tmp . $no;
  }
  else
  {
    $no;
  }
} # end of StrNo


sub VarsList()
{
  my (@pgm) = @_;
  my (%varused, $meanof, $varindex);

  # Note: will define global hash table %VarIndexed

  foreach (@pgm)
  {
    s/#.*//;

    undef $meanof;

    if (/<-\s*mean\(([^\)]+)\[\s*\]\s*\)/)
    {
      $meanof = $1 unless $' =~ /\S/;
    }

    if (/<-/ || /~/)
    {
      $_ = $`;
      undef $varindex;

      if (/[\[\(]/ && $& eq "(")
      {
        s/.*\(//;
        s/\).*//;
      }

      if (/\[(.*)\]/)
      {
        ($_, $varindex) = ($`, $1);
        $varindex =~ s/ //g;
        $varindex =~ s/\./_/g;
        undef $varindex if $varindex =~ /\W/;
        $varindex =~ s/_/\./g if defined $varindex;
      }

      s/\s*(.*)/\1/g; # ltrim
      s/\s+$//;       # rtrim

      $varused{$_} = 1;
      $VarIndexed{$_}{$varindex} = 1 if $varindex;


      if ($meanof)
      {
        $_ .= "\tmean of\t$meanof";
        $varused{$_} = 1;
      }      
    }
  }

  keys %varused;
} # end of VarsList


sub Writable()
{
  my ($file) = @_;
  my ($dir) = Win32::GetFullPathName($file);

  -w $dir;
} # end of Writable


sub WriteAdaptedModel()
{
  my ($model0, $newmodel, @kw) = @_;

  open(SRC, $model0) || die "Cannot write src model from file $model0\nProgram aborted.\n";
  open(NEW, ">$newmodel") || die "Cannot write adpated WinBUGS model to file $newmodel\nProgram aborted.\n";

  while (<SRC>)
  {
    if (/\\/)
    {
      my $keep = 0;
      foreach my $kw (@kw)
      {
        if (/\\$kw\\/i)
        {
          $keep = 1;
          last;
        }
      }

      next unless $keep;
      s/\s*#[^#]*\\//;
    }

    print NEW;
  }

  close NEW;
  close SRC;
} # end of WriteAdaptedModel


sub WriteScript()
{
  my (@instructions) = @_;

  my ($askednchains, $CodaAllNodes, $codalog, $codamonitoring, $codapath, $islist,
      $model, $ninitschains, $noOdc, $o, $odc, $parms, $rmScientificNotation, $saveas, $script, $srcmodel,
      $tmpl, $vars2drop, $wblog,
      @clear, @coda, @datafiles, @density, @die, @history, @ignoredinstructions, @lines2keepkwds, @pgm, @set, @stats,
      %altname, %cleared, %coda,
      %ComputeExp, %computeexpindexed, %computeexpstar,
      %ComputeInvlogit, %computeinvlogitindexed, %computeinvlogitstar,
      %cstvalue, %dbern, %donotmonitor,
      %donotmonitorindexed, %donotplot, %donotreportstats, %ForceMonitoring,
      %inits, %initsRnd, %isdata, %isInitsChain, %isInitsFile, %isInitsSrc, %isvariable,
      %mcmc, %parmvalue, %rndmonitor, %structure, %toclear);


  # Global variables:
  # -----------------
  # $scriptno
  # %action
  # %ParmValue
  # %VarIndexed
  
  
  undef %action;     # will be available throughout WriteScript, but undefined at the end of each call
  undef %VarIndexed; # defined through sub VarsList, from required file winbugs.txt

  $scriptno++;


  # ==== Read  arguments ==========================================================================

  # -- 1st line (model and number of iterations) -------

    # or, maybe, a series of arguments, between curly brackets
    while ($instructions[0] =~ /\s*\{(\w+)\}\s*/ && !$`)
    {
      my $parm = $1;
      my $filewithparmvalue = $';
      $filewithparmvalue =~ s/\s+$//; # rtrim
      $parmvalue{$parm} = defined $ParmValue{$parm}{$filewithparmvalue} ? $ParmValue{$parm}{$filewithparmvalue} : &ParmValue($parm, $filewithparmvalue);
      shift @instructions;
    }

    foreach my $parm (keys %parmvalue)
    {
      foreach (@instructions)
      {
        s/\{$parm\}/$parmvalue{$parm}/g;
      }
    }


  $_ = shift @instructions;
  $o = $_;

  ($model, $_) = &ExtractFilename($_);

  if (-f $model && -T _)
  {
    $model = Win32::GetFullPathName($model);
    $model =~ s{\\}{/}g;
  }
  else
  {
    push(@die, "No valid model file name given in line below", $o, "Sorry", "");
  }


  if (/\S/)
  {
    s/\s+//g;
    my ($k1, $k2, $k3, $k4);

    if (/(\d+)(k?)\/(\d+)(k?)/ && !$')
    {
      ($mcmc{burnin}, $k1, $mcmc{gibbs}, $k2) = ($1, $2, $3, $4);
      $mcmc{burnin} *= 1000 if $k1 =~ /k/i;
      $mcmc{gibbs}  *= 1000 if $k2 =~ /k/i;
    }
    elsif (/(\d+)(k?)\/(\d+)(k?)\/(\d+)(k?)/ && !$')
    {
      ($mcmc{burnin}, $k1, $mcmc{gibbs}, $k2, $mcmc{thin}, $k3) = ($1, $2, $3, $4, $5, $6);
      $mcmc{burnin} *= 1000 if $k1 =~ /k/i;
      $mcmc{gibbs}  *= 1000 if $k2 =~ /k/i;
      $mcmc{thin}   *= 1000 if $k3 =~ /k/i;
    }
    elsif (/(\d+)(k?)\/(\d+)(k?)\/(\d*)(k?)\/(\d+)(k?)/ && !$')
    {
      ($mcmc{burnin}, $k1, $mcmc{gibbs}, $k2, $mcmc{thin}, $k3, $mcmc{clear}, $k4) = ($1, $2, $3, $4, $5, $6, $7, $8);
      $mcmc{burnin} *= 1000 if $k1 =~ /k/i;
      $mcmc{gibbs}  *= 1000 if $k2 =~ /k/i;
      $mcmc{thin}   *= 1000 if $k3 =~ /k/i;
      $mcmc{clear}  *= 1000 if $k4 =~ /k/i;
    }
  }


  # --- rest of arguments -------------------------

  while (@instructions)
  {
    my $topic;

    $_ = shift @instructions;
    $o = $_;
    s/\s*(.*)/\1/g; # ltrim

    if (substr($_, 0, 1) eq "-")
    {
      /\-(\S*)\s*/;
      ($topic, $_) = (lc $1, $');

      $topic .= ":1" if $topic eq "i";
    }
    else
    {
      push(@die, "Line below does not start with a flag", $o, "Line ignored. Sorry.");
      next;
    }


    # ----


    if ($topic eq "m0")
    {
      die "Please give only one src model file name after '-m0' option. Program aborted.\n" if $srcmodel;

      ($srcmodel, $_) = &ExtractFilename($_);

      if (-f $srcmodel && -T _)
      {
        $srcmodel = Win32::GetFullPathName($srcmodel);
        $srcmodel =~ s{\\}{/}g;
      }
      else
      {
        push(@die, "No valid model file name given in line below", $o, "Sorry", "");
      }

      push(@lines2keepkwds, split);
    }
    elsif ($topic eq "d")
    {
      while (/\S/)
      {
        my $tmp;
        ($tmp, $_) = &ExtractFilename($_);

        if (-f $tmp && -T _)
        {
          $tmp = Win32::GetFullPathName($tmp);
          $tmp =~ s{\\}{/}g;
          push(@datafiles, $tmp);
        }
        else
        {
          push(@die, "$tmp: not a valid (data) file.")
        }
      }
    }
    elsif ($topic =~ /i\:(\d+)/ && !$` && !$')
    {
      my $ch = $1;
      $isInitsChain{$ch} = 1;
      $ninitschains = $ch if $ch > $ninitschains;

      my ($tmp, @tmp);

      while (/"([^"]*)"\s*/)
      {
        push(@tmp, $1);
        $tmp .= $`;
        $_ = $';
      }

      $tmp .= $_;
      $_ = $tmp;
      

      foreach my $tmp (@tmp)
      {
        if (-f $tmp && -T _)
        {
          if (&IsListFile($tmp))
          {
            $tmp = Win32::GetFullPathName($tmp);
            $isInitsFile{$ch}{$tmp} = 1;
          }
          else
          {
            $tmp = Win32::GetFullPathName($tmp);
            $isInitsSrc{$ch}{$tmp} = 1;
          }
        }
        else
        {
          push(@die, "$tmp: not a valid (inits) file name.");
        }
      }


      foreach (split)
      {
        if (!-e)
        {
          if (/=\s*dbern\(/)
          {
            my ($node, $key, $tmp) = ($`, "", $');
            my $value;

            if ($tmp =~ /([^=]*)=([^\)]*)\)/)
            {
              ($key, $value, $tmp) = ($1, $2, $');
            }
            elsif ($tmp =~ /\)/)
            {
              ($key, $value, $tmp) = ("p", $`, $');
            }

            push(@die, "$_: incorrect keyword on inits line -- dbern must be used with either p= or logitp=") unless $dbernKey{$key};
            $key = $dbernKey{$key};
            
            $dbern{$node}{p} = $key eq "p" ? $value : &InvLogit($value);
            $dbern{$node}{size} = $1 if $tmp =~ /\{([^\}]*)\}/;
          }
          elsif (/[=~]/)
          {
            my ($var, $matchtype, $value) = ($`, $&, $');
            $inits{$ch}{$var} = $value;
            $initsRnd{$ch}{$var} = 1 if $matchtype eq "~";
          }

          else
          {
            push(@die, "$_: invalid syntax (for inits value).");
          }
        }
        elsif (-T)
        {
          if (&IsListFile($_))
          {
            $_ = Win32::GetFullPathName($_);
            $isInitsFile{$ch}{$_} = 1;
          }
          else
          {
            $_ = Win32::GetFullPathName($_);
            $isInitsSrc{$ch}{$_} = 1;
          }
        }
        else
        {
          push(@die, "$_: not a valid (inits) file name.");
        }
      }
    }
    elsif ($topic eq "dnm")
    {
      s/,/ /g;

      foreach (split)
      {
        if (substr($_, 0, 2) eq "*[")
        {
          next unless /\[(.*)\]/;
          $donotmonitorindexed{$1} = 1;
        }
        elsif (/\*/)
        {
          s/\./\\\./g;
          s/\*/\.\*/g;
          $action{$topic}{$_} = 1;
        }
        elsif (/\{(.*)\}/)
        {
          $rndmonitor{$`} = $1;
        }
        else
        {
          $action{$topic}{$_} = 1;
        }
      }
    }
    elsif ($topic eq "dnp")
    {
      s/,/ /g;

      foreach (split)
      {
        if (/\*/)
        {
          s/\./\\\./g;
          s/\*/\.\*/g;
          $action{$topic}{$_} = 1;
        }
        else
        {
          $donotplot{$_} = 1;
        }
      }
    }
    elsif ($topic eq "dnrs")
    {
      s/,/ /g;

      foreach (split)
      {
        if (/\*/)
        {
          s/\./\\\./g;
          s/\*/\.\*/g;
          $action{$topic}{$_} = 1;
        }
        else
        {
          $donotreportstats{$_} = 1;
        }
      }
    }
    elsif ($topic eq "exp" || $topic eq "or")
    {
      s{\s*([\/\*])\s*}{\1}g;
      s/,/ /g;

      foreach (split)
      {
        my $altname;

        ($_, $altname) = ($`, $') if /\-\>/;

        if (substr($_, 0, 2) eq "*[")
        {
          next unless /\[(.*)\]/;
          $computeexpindexed{$1} = 1;
        }
        elsif (substr($_, -1) eq "*")
        {
          $_ = substr($_, 0, -1);
          $computeexpstar{$_} = length;
        }
        else
        {
          $ComputeExp{$_} = 1;
        }

        $altname{$_} = $altname if defined $altname;
      }
    }
    elsif ($topic eq "inv.logit")
    {
      s/,/ /g;

      foreach (split)
      {
        my $altname;

        ($_, $altname) = ($`, $') if /\-\>/;


        if (substr($_, 0, 2) eq "*[")
        {
          next unless /\[(.*)\]/;
          $computeinvlogitindexed{$1} = 1;
        }
        elsif (substr($_, -1) eq "*")
        {
          $_ = substr($_, 0, -1);
          $computeinvlogitstar{$_} = length;
        }
        else
        {
          $ComputeInvlogit{$_} = 1;
        }

        $altname{$_} = $altname if defined $altname;
      }
    }
    elsif ($topic eq "m")
    {
      foreach (split)
      {
        $ForceMonitoring{$_} = 1;
      }
    }
    elsif ($topic eq "nch")
    {
      s/\s+//g;
      $askednchains = $_;
    }
    elsif ($topic eq "noodc")
    {
      $noOdc = 1;
    }
    elsif ($topic eq "l" || $topic eq "o")
    {
      ($saveas) = &ExtractFilename($_);
    }
    elsif ($topic eq "p")
    {
      s/^\s+|\s+$//g; # lrtrim
      s/\s*=\s*/=/g;
      s/\s+/ /g;
      $parms .= $_ . "\n";
    }
    elsif ($topic eq "s")
    {
      ($script) = &ExtractFilename($_);

      $script = Win32::GetFullPathName($script);
      push(@die, "Cannot write script to file $script") unless &Writable($script);
    }
    elsif ($topic eq "structure")
    {
      my ($tmp, @tmp) = split;
      $structure{$tmp} = join(",", @tmp);
    }
    elsif ($topic eq "coda")
    {
      $codamonitoring = 1;

      s/,/ /g;

      foreach (split)
      {
        if ($_ eq "*")
        {
          $CodaAllNodes = 1;
        }
        elsif (/\*/)
        {
          s/\./\\\./g;
          s/\*/\.\*/g;
          $action{$topic}{$_} = 1;
        }
        else
        {
          $coda{$_} = 1;
        }
      }
    }
    elsif ($topic eq "coda>")
    {
      ($codapath) = &ExtractFilename($_);
    }
    elsif ($topic eq "x")
    {
      s/,/ /g;
      $vars2drop = $_ if /\S/;
    }
    elsif ($topic eq "rmsn")
    {
      $rmScientificNotation = 1;
    }
    else
    {
      push(@ignoredinstructions, $_);
    }
  }


  # --- Define number of chains to run ----


  my $nch = $askednchains;
  $nch = $ninitschains if $ninitschains > $nch;
  $nch = 1 unless $nch;


  # --- Prepare model if it has to be prepared from src model (-m0 option line) ---


  unless ($model)
  {
    if ($#ignoredinstructions == 0)
    {
      $model = shift @ignoredinstructions;
      undef @ignoredinstructions;
      $model = Win32::GetFullPathName($model);
      $model =~ s{\\}{/}g;
    }
  }

  &WriteAdaptedModel($srcmodel, $model, @lines2keepkwds) if $srcmodel;


  # --- Read model ---


  push(@die, "Cannot find file [model] $model") unless -e $model;
  push(@die, "Cannot read model in $model") unless -r $model;
  open(TMP, $model);
  while (<TMP>)
  {
    chomp;
    s/\t+/ /g;
    push(@pgm, $_);
  }
  close TMP;


  # --- Read data and extract data varnames ---


  foreach my $datafile (@datafiles)
  {
    my ($tmp, @tmp);
    &RmScientificNotationInWinBUGSDatafile($datafile) if $rmScientificNotation;

    open(TMP, $datafile);

    while (<TMP>)
    {
      s/#.*//; # remove comments
      next unless /\S/;

      s/\[[^\]]*\]//g; # remove brackets
      /\s*(\S{1,5})/;
      $islist = $1 eq "list(" ? 1 : 0;

      last;
    }

    unless ($islist)
    {
      $tmp = $_;
      @tmp = (<TMP>);
    }

    close TMP;


    if ($islist)
    {
      @tmp = &ListVarnamesAndOrConstants($datafile, 1, 1, 1);
    }
    else
    {
      # make sure the file ends with an END line
      my $eof;

      while (@tmp)
      {
        $_ = pop @tmp;
        s/#.*//;
        next unless /\S/;

        /\s*(\S+)/;
        $eof = 1 if $1 eq "END";
        last;
      }

      push(@die, "$datafile: rectangular data file does not end with END\nSorry\n") unless $eof;

      # read varnames from header

      @tmp = split(" ", $tmp);
    }


    foreach (@tmp)
    {
      if (/=/)
      {
        my $cstvalue;
        ($_, $cstvalue) = ($`, $');
        $cstvalue{$_} = $cstvalue;
      }

      $isdata{$_} = 1;
    }
  }


  # --- Get full list of variables used in model ---

  my @vars = &VarsList(@pgm);
  

  # Eliminate nodes that are means of data variables

  foreach (@vars)
  {
    next unless /\tmean of\t/;
    my ($var, $meanof) = ($`, $');

    $donotmonitor{$var} = 1 if $isdata{$meanof};
  }


  @vars = &NoTauWhenSD(@vars);


  # --- Drop data variables from (monitored) variable list ---

  my @tmp = sort byVarNamesCaseInsensitive @vars;
  undef @vars;

  foreach (@tmp)
  {
    my $donotmonitor;
    
    if ($isdata{$_})
    {
      next unless $ForceMonitoring{$_};
    }
    else
    {
      next if /\t/ || (($donotmonitor{$_} || &Star("dnm", $_)) && !$ForceMonitoring{$_});
    }

    my $monitor = 1;


    unless ($ForceMonitoring{$_})
    {
      foreach my $dnmi (keys %donotmonitorindexed)
      {
        if ($VarIndexed{$_}{$dnmi})
        {
          $monitor = 0;
          last;
        }
      }
    }

    next unless $monitor;
    my @tmpvars;
    

    if ($rndmonitor{$_} =~ /\\/)
    {
      my ($dim, $m) = ($`, $');
      $dim = $cstvalue{$dim} if $dim =~ /\D/;
      $m = $dim if $m > $dim;
      @tmpvars = sort {$a <=> $b} &RndIntegers($m, 0, $dim);
      foreach my $j (@tmpvars)
      {
        $j = $_ . "[" . $j . "]";
      }
    }
    else
    {
      @tmpvars = ($_);
    }

    foreach my $tmpv (@tmpvars)
    {
      push(@vars, $tmpv);
      $tmpv =~ s/\[.*//;
      $isvariable{$tmpv} = 1;
    }


    # ---

    foreach my $l (values %computeexpstar)
    {
      my $tmp = substr($_, 0, $l);
      if ($computeexpstar{$tmp})
      {
        $ComputeExp{$_} = 1;
        last;
      }
    }


    foreach my $index (keys %computeexpindexed)
    {
      if ($VarIndexed{$_}{$index})
      {
        $ComputeExp{$_} = 1;
        last;
      }
    }

    # ---

    foreach my $l (values %computeinvlogitstar)
    {
      my $tmp = substr($_, 0, $l);
      if ($computeinvlogitstar{$tmp})
      {
        $ComputeInvlogit{$_} = 1;
        last;
      }
    }


    foreach my $index (keys %computeinvlogitindexed)
    {
      if ($VarIndexed{$_}{$index})
      {
        $ComputeInvlogit{$_} = 1;
        last;
      }
    }
  }


  # --- Define list of nodes to include in coda statements -----------------------

  if (keys %coda)
  {
    my @tmp = keys %isvariable;

    foreach my $coda (@tmp)
    {
      if ($coda{$coda} || &Star("coda", $coda))
      {
        push(@coda, $coda);
      }
    }

    @coda = sort {length($b) <=> length($a)} @coda;
    $tmpl = length($coda[0]);

    @coda = sort byVarNamesCaseInsensitive @coda;
  }


  # --- Write WinBUGS script -----------------------------------------------------


  my $printscript = $saveas || $codamonitoring;


  if ($saveas)
  {
    my $logdir;
    ($logdir, $saveas) = Win32::GetFullPathName($saveas);

    $logdir =~ s{\\}{/}g;
    chop $logdir;

    if ($saveas =~ /(.*)\./)
    {
      my $ext = lc $';
      $saveas = $1 if $ext eq "txt" || $ext eq "odc";
    }

    $odc   = "$logdir/$saveas.odc";
    $wblog = "$logdir/$saveas-WinBUGSlog.txt";
  }


  if ($codamonitoring)
  {
    if ($codapath)
    {
      my @tmp = Win32::GetFullPathName($codapath);

      push(@die, "-coda> $tmp[0] not a directory") unless -d $tmp[0];

      $codalog = join("", @tmp);
      $codalog =~ s{\\}{/}g;
    }
    else
    {
      push(@die, "CODA output file(s) location not specified [through -coda> option]") ;
    }
  }


  my @script = ("display(log)",
                "check('$model')");

  foreach my $datafile (@datafiles)
  {
    push(@script, "data('" . &SlashFullPathName($datafile) . "')");
  }

  if (@datafiles && $vars2drop)
  {
    push(@script, "# RunWinBUGSScript will drop the following variable(s) from file(s) above:", "# $vars2drop");
  }


  if ($parms)
  {
    chomp $parms;
    $parms =~ s/\s+/, /g;

    my ($writeNew, $tmp) = &ParmsFileName($parms);

    if ($writeNew)
    {
      my @tmp = &ShortLines($Width{parms}, "list(", $parms, ")");
      open(TMP, ">$tmp");
      foreach (@tmp)
      {
        print TMP $_ . "\n";
      }
      close TMP;
    }

    push(@script, "data('" . &SlashFullPathName($tmp) . "')");
  }


  push(@script, "compile($nch)");


  # --- Inits ---------------------------------------------------------------------------------


  if ($printscript)
  {
    my (%altInits, %altInitsRnd);
    my @chains = sort {$a <=> $b} keys %isInitsChain;

    foreach my $ch (@chains)
    {
      my @inits;

      # --- Read inits src files, and take values defined therein as initial values, for nodes yet initialized ------

      my @initssrcfiles = keys %{$isInitsSrc{$ch}};
      

      foreach my $srcfile (@initssrcfiles)
      {
        my $tmp;
      
        open(TMP, $srcfile);
        my @tmp = (<TMP>);
        close TMP;
        

        foreach (@tmp)
        {
          chomp;
          s/#.*//;
          s/,/, /g;
        }
        
        
        # Remove leading "list(" declaration
        
        while (@tmp)
        {
          $tmp = shift @tmp;
          next unless $tmp =~ /\S/;
          $tmp = $' if $tmp =~ /list\(\s*/;
          unshift(@tmp, $tmp);
          last;
        }
        
        # Read node values
        
        my @nodeValues = &NodeValues(@tmp);

        while (@nodeValues)
        {
          my ($node, $matchtype, $value) = splice(@nodeValues, 0, 3);
          $altInits{$ch}{$node} = $value;
          $altInitsRnd{$ch}{$node} = 1 if $matchtype eq "~";
        }
      }


      # --- Add inits values read from inits src files, if necessary --------------------------------------------------

      my @nodes = keys %{$altInits{$ch}};

      foreach my $node (@nodes)
      {
        # next unless $isvariable{$node};
        next if defined $inits{$ch}{$node};
        $inits{$ch}{$node} = $altInits{$ch}{$node};
        $initsRnd{$ch}{$node} = 1 if $altInitsRnd{$ch}{$node};
      }


      # ---------------------------------------------------------------------------------------------------------------
      @nodes = sort byVarNamesCaseInsensitive keys %{$inits{$ch}};

      foreach my $varname (@nodes)
      {
        my $value = $inits{$ch}{$varname};
        $value =~ s/ //g;
        $value =~ s/,+/,/g;

        if ($value =~ /(.*)\{(.+)\}/)
        {
          my $nrep;
          ($value, $nrep) = ($1, $2);

          my $firstna = 0;
          ($nrep, $firstna) = (substr($nrep, 0, -1), 1) if substr($nrep, -1) eq "*";

          my $isinteger = &IsInteger($nrep);
          push(@die, "Constant $nrep (in initial values) not found in data files") unless exists $cstvalue{$nrep} || $isinteger;

          if ($isinteger)
          {
            $value = &Rep($value, $nrep, $firstna);
          }
          elsif ($value =~ /\:/)
          {
            $value = &RndValues($value, $cstvalue{$nrep});
          }
          else
          {
            $value = &Rep($value, $cstvalue{$nrep}, $firstna);
          }
        }
        elsif ($initsRnd{$ch}{$varname} && $inits{$ch}{$varname} =~ /\s*n\s*\(([^,]*),([^,]*),([^\)]*)\)/i)
        {
          undef $value;
          my ($rnormMu, $rnormSD, $rnormL) = ($1, $2, $3);

          foreach ($rnormMu, $rnormSD, $rnormL)
          {
            s/\s+//g;
          }

          my $rnormM = @RndZ; # number of components in @RndZ
          $value = "structure(.Data=" if $structure{$varname};
          $value .= "c(";
          $rnormL = $cstvalue{$rnormL} if defined $cstvalue{$rnormL};

          foreach (1..$rnormL)
          {
            my $tmp = int(rand($rnormM));
            $tmp = $rnormMu + $rnormSD * $RndZ[$tmp];
            $value .= $tmp . ",";
          }
          chop $value;
          $value .= ")";
          $value .= ", .Dim=c(" . $structure{$varname} . "))" if $structure{$varname};
        }

        $value =~ s/,\s*/, /g;
        my @tmp = &ShortLines($Width{inits}, "  $varname = $value");
        $tmp[$#tmp] .= ",";
        push(@inits, @tmp);
      }


      # Add dbern() inits, if any
      
      my @dbernVars = sort {lc $a cmp lc $b} keys %dbern;
      if (@dbernVars)
      {
        my (@dbernDataVars, %isdbernChanceNode, %isdbernDataVar);

        foreach (@dbernVars)
        {
          if ($isdata{$_})
          {
            $isdbernDataVar{$_} = 1;
          }
          else
          {
            $isdbernChanceNode{$_} = 1;
          }
        }
        @dbernDataVars = keys %isdbernDataVar;
        
        foreach my $file (@datafiles)
        {
          last unless @dbernDataVars;

          my %tmp = &ListValues($file, @dbernDataVars);
          foreach my $var (keys %tmp)
          {
            my $values = $tmp{$var};
            $values =~ s/\).*//;
            $values =~ s/.*\(//;
            $values =~ s/,/ /g;
            $values =~ s/\s+/ /g;
            $values =~ s/^\s+|\s+$//g; # lrtrim
            
            my @varInits;
            my @values = split(" ", $values);
            foreach my $value (@values)
            {
              my $init = $value eq "NA" ? &dbern($dbern{$var}{p}) : "NA";
              push(@varInits, $init);
            }
            my $varInits = "  $var = c(" . join(", ", @varInits) . "),";
            push(@inits, $varInits);
            
            delete $isdbernDataVar{$var};
          }
          
          @dbernDataVars = keys %isdbernDataVar;
        }
        
        
        my @dbernChanceNodes = sort {lc $a cmp lc $b} keys %isdbernChanceNode;
        foreach my $node (@dbernChanceNodes)
        {
          my $size = $dbern{$node}{size};
          $size = $cstvalue{$size} if $size =~ /\D/;
          my @varInits;
          foreach (1..$size)
          {
            push(@varInits, &dbern($dbern{$node}{p}));
          }
          my $varInits = "  $node = c(" . join(", ", @varInits) . "),";
          push(@inits, $varInits);
        }
      }
      
      
      # --- Write additional inits file, if necessary  --------------------------------------------------------------
      
      next unless @inits;

      my $initsfile = &FirstAvailableFilenameWithNumerixExtension($Tmp{inits}, "txt", 4);
      $initsfile = Win32::GetFullPathName($initsfile);
      push(@die, "Cannot write WinBUGS inits values to temporary file $initsfile") unless &Writable($initsfile);

      @inits = &ShortLines($Width{inits}, @inits);
      
      
      $inits[$#inits] = $1 if $inits[$#inits] =~ /(.*[^,]),+\s*/ && !$'; # remove trailing comma on inits last line
      push(@inits, ")");
      unshift(@inits, "# This WinBUGS inits file was written for script $script", "",  "list(");
      

      open(INITS, ">$initsfile");
      foreach (@inits)
      {
        print INITS $_ .  "\n";
      }
      close INITS;

      $isInitsFile{$ch}{$initsfile} = 1;
    }


    # --- Add inits files to WinBUGS code ---------------------------

    @chains = sort {$a <=> $b} keys %isInitsFile;

    foreach my $ch (@chains)
    {
      my @initsfiles = sort {lc $a cmp lc $b} keys %{$isInitsFile{$ch}};

      foreach my $initsfile (@initsfiles)
      {
        &RmScientificNotationInWinBUGSDatafile($initsfile) if $rmScientificNotation;

        $initsfile =~ s{\\}{/}g;
        push(@script, "inits(" . $ch . ", '" . $initsfile . "')");
      }
    }


    push(@script, "gen.inits()");
  }

  # ----------------------------------------------------------------------------

  push(@script, "update($mcmc{burnin})") if $mcmc{burnin};


  if (!$saveas && !$CodaAllNodes)
  {
    foreach (@coda)
    {
      push(@set, "set($_)");
      $toclear{$_} = 1 if $mcmc{clear};
    }
  }
  else
  {
    foreach (@vars)
    {
      push(@set, "set($_)");
      $toclear{$_} = 1 if $mcmc{clear};
    }
  }

  push(@script, "thin.updater($mcmc{thin})") if $mcmc{thin};


  @vars = sort byVarNamesCaseInsensitive keys %isvariable; # thus, if only z[4], z[5], z[6] are monited, we will print density(z) only once
                                                           # rather than once for each z item monitored

  if ($saveas)
  {
    my $ReportStatsOnAll = 1;
    undef $ReportStatsOnAll if keys %donotreportstats;
    undef $ReportStatsOnAll if keys %{$action{dnrs}};


    unless ($ReportStatsOnAll)
    {
      foreach (@vars)
      {
        if ($donotreportstats{$_} || &Star("dnrs", $_))
        {
          push(@stats, "clear($_)");
          $cleared{$_} = 1;
        }
      }
    }
    push(@stats, "stats(*)");

  
  
    unless ($noOdc)
    {
      foreach (@vars)
      {
        push(@density, "density($_)") unless $donotplot{$_} || &Star("dnp", $_);
      }

      foreach (@vars)
      {
        push(@history, "history($_)") unless $donotplot{$_} || &Star("dnp", $_);
      }
    }
    
    
    my @toclear = sort {lc $a cmp lc $b} keys %toclear;
    foreach (@toclear)
    {
      push(@clear, "clear($_)") unless $cleared{$_};
    }
    
    
    # Prepare coda() code

    my @codascript;
    $codalog = $1 if $codalog =~ /(.*)\.txt/i && !$'; # remove the .txt extension if any


    if ($CodaAllNodes)
    {
      push(@codascript, "coda(*, '$codalog<>')");
    }
    elsif (@coda)
    {
      my ($suffx, $suffxk);

      foreach my $coda (@coda)
      {
        $suffx = "_" . &MyChr(++$suffxk) if $#coda;
        my $spaces = " " x ($tmpl - length($coda));
        push(@codascript, "coda($spaces$coda, '$codalog$suffx<>')");
      }
    }
    
    # Loop to run the desired number of iterations
    
    my $niterLeft = $mcmc{gibbs};
    my $blockNo = 0;

    while ($niterLeft)
    {
      my $niter = $niterLeft;
      $niter = $mcmc{clear} if $mcmc{clear} && $niter > $mcmc{clear};
      $niterLeft -= $niter;
      
      push(@script, @set, "update($niter)");
                    
      if ($ReportStatsOnAll)
      {
        push(@script, @stats);
        push(@script, @density) if !$mcmc{clear} || $mcmc{clear} >= $mcmc{gibbs};
        push(@script, @history);
      }
      else
      {
        push(@script, @density) if !$mcmc{clear} || $mcmc{clear} >= $mcmc{gibbs};
        push(@script, @history);
        push(@script, @stats);
      }
      
      if (@codascript)
      {
        my @tmp = @codascript;
        $blockNo++;
        my $index = $niterLeft > 0 || $blockNo > 1 ? "-$blockNo" : "";
        
        foreach (@tmp)
        {
          s/\<\>/$index/;
        }
        push(@script, @tmp);
      }
                    
      push(@script, @clear);
                    

    }
    
    push(@script, "save('$odc')") unless $noOdc;
    push(@script, "save('" . &SlashFullPathName($wblog) . "')");
  }


  # --- Issue error message (s) or write script ---------------------------------


  push(@die, "neither -o nor -coda option was used (thus no output/coda file mentionned)") unless $printscript;
  push(@die, "script file name was not given (through -s option)") unless $script;


  if (@die)
  {
    unlink $script;
    unshift(@die, "Calling dir:" . Win32::GetCwd(), "Ignored instructions: " . join(" ", @ignoredinstructions), "") unless $model;

    print STDERR "\nScript # $scriptno ignored due to:\n\t(script=$script)\n";
    foreach (@die)
    {
      chomp;
      print STDERR "\t" . $_ . "\n";
    }
    print STDERR "\n";
  }
  else
  {
    push(@script, "quit()");

    @tmp = sort {lc $a cmp lc $b} keys %ComputeExp;
    if (@tmp)
    {
      foreach (@tmp)
      {
        $_ .= "->$altname{$_}" if defined $altname{$_};
      }

      $_ = "# when done, compute stats for exp() of";
      $_ .= ":" if $#tmp;
      $_ = join(" ", $_, @tmp);
      push(@script, $_);
    }

    @tmp = sort {lc $a cmp lc $b} keys %ComputeInvlogit;
    if (@tmp)
    {
      foreach (@tmp)
      {
        $_ .= "->$altname{$_}" if defined $altname{$_};
      }

      $_ = "# when done, compute stats for inv.logit() of";
      $_ .= ":" if $#tmp;
      $_ = join(" ", $_, @tmp);
      push(@script, $_);
    }


    open(SCRIPT, ">$script");
    foreach (@script)
    {
      chomp;
      print SCRIPT $_ . "\n";
    }
    close SCRIPT;

    push(@WrittenScripts, $script);
  }
} # end of WriteScript
