# Consensus-Based Sample Size - Binomial proportion
#   Version 0.10 (November 2014) 

# --- Do not edit this file ----------------------------------------------------
                                                  

# ******************************************************************************
# Functions for sample size calculation with criteria involving HPD limits
#                                                                                   
# ******************************************************************************


ss.cons.binom.worst.hpdlimits <- function(accepted.pdiff, prior1, prior2,
                                                    level=0.95, n.max=1e7,
                                                    fast.approx=T)
{
  # prior1 and prior2 must be lists with dimensions alpha & beta
  tmp <- .ss.cons.binom.check.list(prior1, "prior1")
  tmp <- .ss.cons.binom.check.list(prior2, "prior2")
  
  
  nStart.consensus.binom <- function(alphas, betas, accepted.pdiff, level)
  {
    # To find starting value for sample size, based on assumption
    # that sample size required will be large, assuming that
    # worst outcome (in terms of posterior quantiles differences)
    # will be with either no success or no failure in the observed data,
    # and using normal approximation to posterior distributions (betas with
    # large value for sample size equivalence [alpha+beta])

    z <- qnorm((1+level)/2)
    tau <- alphas + betas

    solns <- numeric(0)
    for (g1 in c(-1,1))
    {
      eta <- alphas + g1 * z * sqrt(alphas)
      rho <- alphas + g1 * z * sqrt(betas)

      for (g2 in c(-1,1))
      {
        A <- g2 * accepted.pdiff
        B <- A * sum(tau) + diff(eta)
        C <- eta[2]*tau[1] - eta[1]*tau[2] + A * prod(tau)
        
        delta <- B*B - 4*A*C
        if (delta >= 0)
        {
          solns0 <- (-B + c(-1,1)*sqrt(B*B-4*A*C))/(2*A)
          solns <- c(solns, solns0)
        }

        B <- A * sum(tau) + diff(rho) - diff(tau)
        C <- rho[2]*tau[1] - rho[1]*tau[2] + A * prod(tau)
        
        delta <- B*B - 4*A*C
        if (delta >= 0)
        {
          solns0 <- (-B + c(-1,1)*sqrt(B*B-4*A*C))/(2*A)
          solns <- c(solns, solns0)
        }
      }
    }

    ceiling(max(solns, na.rm=T))
  } # end of nStart.consensus.binom

  alphas  <- as.double(c(prior1$alpha, prior2$alpha))
  betas   <- as.double(c(prior1$beta,  prior2$beta))
  n.start <- nStart.consensus.binom(alphas, betas, accepted.pdiff, level)

  tmp <- .ss.cons.binom.hpdlimits(accepted.pdiff, prior1, prior2, level, fast.approx,
                                            n.start, n.max, target=accepted.pdiff, return.worst=T)

  tmp <- c(tmp, list(accepted.pdiff=accepted.pdiff, prior1=prior1, prior2=prior2, level=level, n.max=n.max, fast.approx=fast.approx))
  tmp
} # end of ss.cons.binom.worst.hpdlimits 


ss.cons.binom.avg.hpdlimits <- function(accepted.pdiff, prior1, prior2, prior1.mixture.wt=0.5,
                                                  level=0.95, n.start=1000, n.max=1e7)
{
  # prior1 and prior2 must be lists with dimensions alpha & beta
  tmp <- .ss.cons.binom.check.list(prior1, "prior1")
  tmp <- .ss.cons.binom.check.list(prior2, "prior2")
  
  tmp <- .ss.cons.binom.hpdlimits(accepted.pdiff, prior1, prior2, level, fast.approx=F, 
                                            n.start, n.max, target=accepted.pdiff, prior1.mixture.wt=prior1.mixture.wt)

  tmp <- c(tmp, list(accepted.pdiff=accepted.pdiff, prior1=prior1, prior2=prior2, prior1.mixture.wt=prior1.mixture.wt, level=level, n.start=n.start, n.max=n.max))
  tmp
} # end of ss.cons.binom.avg.hpdlimits


ss.cons.binom.avg.hpdlimits.mymarg <- function(accepted.pdiff, prior1, prior2, clinical.prior,
                                                         level=0.95, n.start=1000, n.max=1e7)
{
  # prior1, prior2 and clinical.prior must be lists with dimensions alpha & beta
  tmp <- .ss.cons.binom.check.list(prior1, "prior1")
  tmp <- .ss.cons.binom.check.list(prior2, "prior2")
  tmp <- .ss.cons.binom.check.list(clinical.prior, "clinical.prior")

  
  tmp <- .ss.cons.binom.hpdlimits(accepted.pdiff, prior1, prior2, level, fast.approx=F,
                                            n.start, n.max, clinical.prior=clinical.prior, target=accepted.pdiff)

  tmp <- c(tmp, list(accepted.pdiff=accepted.pdiff, prior1=prior1, prior2=prior2, clinical.prior=clinical.prior, level=level, n.start=n.start, n.max=n.max))
  tmp
} # end of ss.cons.binom.avg.hpdlimits.mymarg


ss.cons.binom.avg.hpdlimits.bothmarg <- function(accepted.pdiff, prior1, prior2,
                                                           level=0.95, n.start=1000, n.max=1e7)
{
  # prior1 and prior2 must be lists with dimensions alpha & beta
  tmp <- .ss.cons.binom.check.list(prior1, "prior1")
  tmp <- .ss.cons.binom.check.list(prior2, "prior2")
  
  
  tmp <- .ss.cons.binom.hpdlimits(accepted.pdiff, prior1, prior2, level, fast.approx=F,
                                            n.start, n.max, target=accepted.pdiff, both=T)

  tmp <- c(tmp, list(accepted.pdiff=accepted.pdiff, prior1=prior1, prior2=prior2, level=level, n.start=n.start, n.max=n.max))
  tmp
} # end of ss.cons.binom.avg.hpdlimits.bothmarg


ss.cons.binom.prob.hpdlimits <- function(accepted.pdiff, prior1, prior2, prior1.mixture.wt=0.5, prob=0.5,
                                                   level=0.95, n.start=1000, n.max=1e7)
{
  # prior1 and prior2 must be lists with dimensions alpha & beta
  tmp <- .ss.cons.binom.check.list(prior1, "prior1")
  tmp <- .ss.cons.binom.check.list(prior2, "prior2")


  tmp <- .ss.cons.binom.hpdlimits(accepted.pdiff, prior1, prior2, level, fast.approx=F,  
                                            n.start, n.max, target=prob, prior1.mixture.wt=prior1.mixture.wt, return.prob=T, increasing.outcome.with.n=T)

  tmp <- c(tmp, list(accepted.pdiff=accepted.pdiff, prior1=prior1, prior2=prior2, prior1.mixture.wt=prior1.mixture.wt, prob=prob, level=level, n.start=n.start, n.max=n.max))
  tmp
} # end of ss.cons.binom.prob.hpdlimits


ss.cons.binom.prob.hpdlimits.mymarg <- function(accepted.pdiff, prior1, prior2, clinical.prior, prob=0.5, 
                                                          level=0.95, n.start=1000, n.max=1e7)
{
  # prior1, prior2 and clinical.prior must be lists with dimensions alpha & beta
  tmp <- .ss.cons.binom.check.list(prior1, "prior1")
  tmp <- .ss.cons.binom.check.list(prior2, "prior2")
  tmp <- .ss.cons.binom.check.list(clinical.prior, "clinical.prior")
  
  tmp <- .ss.cons.binom.hpdlimits(accepted.pdiff, prior1, prior2, level, fast.approx=F,
                                            n.start, n.max, 
                                            clinical.prior=clinical.prior, target=prob, return.prob=T, increasing.outcome.with.n=T)

  tmp <- c(tmp, list(accepted.pdiff=accepted.pdiff, prior1=prior1, prior2=prior2, clinical.prior=clinical.prior, prob=prob, level=level, n.start=n.start, n.max=n.max))
  tmp
} # end of ss.cons.binom.prob.hpdlimits.mymarg


ss.cons.binom.prob.hpdlimits.bothmarg <- function(accepted.pdiff, prior1, prior2, prob=0.5,
                                                            level=0.95, n.start=1000, n.max=1e7)
{
  # prior1 and prior2 must be lists with dimensions alpha & beta
  tmp <- .ss.cons.binom.check.list(prior1, "prior1")
  tmp <- .ss.cons.binom.check.list(prior2, "prior2")


  tmp <- .ss.cons.binom.hpdlimits(accepted.pdiff, prior1, prior2, level, fast.approx=F, 
                                            n.start, n.max, 
                                            target=prob, both=T, return.prob=T, increasing.outcome.with.n=T)

  tmp <- c(tmp, list(accepted.pdiff=accepted.pdiff, prior1=prior1, prior2=prior2, prob=prob, level=level, n.start=n.start, n.max=n.max))
  tmp
} # end of ss.cons.binom.prob.hpdlimits.bothmarg



# ************************************************************************************************************************************************************
# Functions for sample size calculation with criteria involving fixed quantiles
#
# ************************************************************************************************************************************************************


ss.cons.binom.worst.cdf <- function(cdf.points, accepted.cdf.diff, prior1, prior2,
                                               n.start=1000, n.max=1e7)
{
  # prior1 and prior2 must be lists with dimensions alpha & beta
  tmp <- .ss.cons.binom.check.list(prior1, "prior1")
  tmp <- .ss.cons.binom.check.list(prior2, "prior2")


  tmp <- .ss.cons.binom.cdf(cdf.points, accepted.cdf.diff, prior1, prior2,
                                       n.start, n.max, return.worst=T)

  tmp <- c(tmp, list(cdf.points=cdf.points, accepted.cdf.diff=accepted.cdf.diff, prior1=prior1, prior2=prior2, n.start=n.start, n.max=n.max))
  tmp
} # end of ss.cons.binom.worst.cdf


ss.cons.binom.avg.cdf <- function(cdf.points, accepted.cdf.diff, prior1, prior2, prior1.mixture.wt=0.5,  
                                             n.start=1000, n.max=1e7)
{
  # prior1 and prior2 must be lists with dimensions alpha & beta
  tmp <- .ss.cons.binom.check.list(prior1, "prior1")
  tmp <- .ss.cons.binom.check.list(prior2, "prior2")
  

  tmp <- .ss.cons.binom.cdf(cdf.points, accepted.cdf.diff, prior1, prior2,
                                       n.start, n.max, prior1.mixture.wt=prior1.mixture.wt)

  tmp <- c(tmp, list(cdf.points=cdf.points, accepted.cdf.diff=accepted.cdf.diff, prior1=prior1, prior2=prior2, prior1.mixture.wt=prior1.mixture.wt, n.start=n.start, n.max=n.max))
  tmp
} # end of ss.cons.binom.avg.cdf


ss.cons.binom.avg.cdf.mymarg <- function(cdf.points, accepted.cdf.diff, prior1, prior2, clinical.prior,
                                                    n.start=1000, n.max=1e7)
{
  # prior1 and prior2 must be lists with dimensions alpha & beta
  tmp <- .ss.cons.binom.check.list(prior1, "prior1")
  tmp <- .ss.cons.binom.check.list(prior2, "prior2")
  tmp <- .ss.cons.binom.check.list(clinical.prior, "clinical.prior")
  

  tmp <- .ss.cons.binom.cdf(cdf.points, accepted.cdf.diff, prior1, prior2,
                                       n.start, n.max, clinical.prior=clinical.prior)

  tmp <- c(tmp, list(cdf.points=cdf.points, accepted.cdf.diff=accepted.cdf.diff, prior1=prior1, prior2=prior2, clinical.prior=clinical.prior, n.start=n.start, n.max=n.max))
  tmp
} # end of ss.cons.binom.avg.cdf.mymarg


ss.cons.binom.avg.cdf.bothmarg <- function(cdf.points, accepted.cdf.diff, prior1, prior2,
                                                      n.start=1000, n.max=1e7)
{
  # prior1 and prior2 must be lists with dimensions alpha & beta
  tmp <- .ss.cons.binom.check.list(prior1, "prior1")
  tmp <- .ss.cons.binom.check.list(prior2, "prior2")


  tmp <- .ss.cons.binom.cdf(cdf.points, accepted.cdf.diff, prior1, prior2,
                                       n.start, n.max, both=T)

  tmp <- c(tmp, list(cdf.points=cdf.points, accepted.cdf.diff=accepted.cdf.diff, prior1=prior1, prior2=prior2, n.start=n.start, n.max=n.max))
  tmp
} # end of ss.cons.binom.avg.cdf.bothmarg


ss.cons.binom.prob.cdf <- function(cdf.points, accepted.cdf.diff, prior1, prior2, prior1.mixture.wt=0.5, prob=0.5,
                                              n.start=1000, n.max=1e7)
{
  # prior1 and prior2 must be lists with dimensions alpha & beta
  tmp <- .ss.cons.binom.check.list(prior1, "prior1")
  tmp <- .ss.cons.binom.check.list(prior2, "prior2")


  tmp <- .ss.cons.binom.cdf(cdf.points, accepted.cdf.diff, prior1, prior2,
                                       n.start, n.max, target=prob, prior1.mixture.wt=prior1.mixture.wt, return.prob=T, increasing.outcome.with.n=T)

  tmp <- c(tmp, list(cdf.points=cdf.points, accepted.cdf.diff=accepted.cdf.diff, prior1=prior1, prior2=prior2, prior1.mixture.wt=prior1.mixture.wt, prob=prob, n.start=n.start, n.max=n.max))
  tmp
} # end of ss.cons.binom.prob.cdf


ss.cons.binom.prob.cdf.mymarg <- function(cdf.points, accepted.cdf.diff, prior1, prior2, clinical.prior, prob=0.5,
                                                     n.start=1000, n.max=1e7)
{
  # prior1, prior2 and clinical.prior must be lists with dimensions alpha & beta
  tmp <- .ss.cons.binom.check.list(prior1, "prior1")
  tmp <- .ss.cons.binom.check.list(prior2, "prior2")
  tmp <- .ss.cons.binom.check.list(clinical.prior, "clinical.prior")

  tmp <- .ss.cons.binom.cdf(cdf.points, accepted.cdf.diff, prior1, prior2,
                                       n.start, n.max, clinical.prior=clinical.prior, 
                                       target=prob, return.prob=T, increasing.outcome.with.n=T)

  tmp <- c(tmp, list(cdf.points=cdf.points, accepted.cdf.diff=accepted.cdf.diff, prior1=prior1, prior2=prior2, clinical.prior=clinical.prior, prob=prob, n.start=n.start, n.max=n.max))
  tmp
} # end of ss.cons.binom.prob.cdf.mymarg


ss.cons.binom.prob.cdf.bothmarg <- function(cdf.points, accepted.cdf.diff, prior1, prior2, prob=0.5,
                                                       n.start=1000, n.max=1e7)
{
  # prior1 and prior2 must be lists with dimensions alpha & beta
  tmp <- .ss.cons.binom.check.list(prior1, "prior1")
  tmp <- .ss.cons.binom.check.list(prior2, "prior2")


  tmp <- .ss.cons.binom.cdf(cdf.points, accepted.cdf.diff, prior1, prior2,
                                       n.start, n.max, target=prob, both=T, return.prob=T, increasing.outcome.with.n=T)

  tmp <- c(tmp, list(cdf.points=cdf.points, accepted.cdf.diff=accepted.cdf.diff, prior1=prior1, prior2=prior2, prob=prob, n.start=n.start, n.max=n.max))
  tmp
} # end of ss.cons.binom.prob.cdf.bothmarg


ss.cons.binom.worst.q <- function(quantiles, accepted.pdiff, prior1, prior2,
                                                    n.start=1000, n.max=1e7)
{
  # prior1 and prior2 must be lists with dimensions alpha & beta
  tmp <- .ss.cons.binom.check.list(prior1, "prior1")
  tmp <- .ss.cons.binom.check.list(prior2, "prior2")
  

  tmp <- .ss.cons.binom.q(quantiles, accepted.pdiff, prior1, prior2,
                                            n.start, n.max, return.worst=T)

  tmp <- c(tmp, list(quantiles=quantiles, accepted.pdiff=accepted.pdiff, prior1=prior1, prior2=prior2, n.start=n.start, n.max=n.max))
  tmp
} # end of ss.cons.binom.worst.q


ss.cons.binom.avg.q <- function(quantiles, accepted.pdiff, prior1, prior2, prior1.mixture.wt=0.5, 
                                                  n.start=1000, n.max=1e7)
{
  # prior1 and prior2 must be lists with dimensions alpha & beta
  tmp <- .ss.cons.binom.check.list(prior1, "prior1")
  tmp <- .ss.cons.binom.check.list(prior2, "prior2")
  

  tmp <- .ss.cons.binom.q(quantiles, accepted.pdiff, prior1, prior2,
                                            n.start, n.max, prior1.mixture.wt=prior1.mixture.wt)

  tmp <- c(tmp, list(quantiles=quantiles, accepted.pdiff=accepted.pdiff, prior1=prior1, prior2=prior2, prior1.mixture.wt=prior1.mixture.wt, n.start=n.start, n.max=n.max))
  tmp
} # end of ss.cons.binom.avg.q


ss.cons.binom.avg.q.mymarg <- function(quantiles, accepted.pdiff, prior1, prior2, clinical.prior,
                                                         n.start=1000, n.max=1e7)
{
  # prior1 and prior2 must be lists with dimensions alpha & beta
  tmp <- .ss.cons.binom.check.list(prior1, "prior1")
  tmp <- .ss.cons.binom.check.list(prior2, "prior2")
  tmp <- .ss.cons.binom.check.list(clinical.prior, "clinical.prior")
  

  tmp <- .ss.cons.binom.q(quantiles, accepted.pdiff, prior1, prior2,
                                            n.start, n.max, clinical.prior=clinical.prior)

  tmp <- c(tmp, list(quantiles=quantiles, accepted.pdiff=accepted.pdiff, prior1=prior1, prior2=prior2, clinical.prior=clinical.prior, n.start=n.start, n.max=n.max))
  tmp
} # end of ss.cons.binom.avg.q.mymarg


ss.cons.binom.avg.q.bothmarg <- function(quantiles, accepted.pdiff, prior1, prior2,
                                                           n.start=1000, n.max=1e7)
{
  # prior1 and prior2 must be lists with dimensions alpha & beta
  tmp <- .ss.cons.binom.check.list(prior1, "prior1")
  tmp <- .ss.cons.binom.check.list(prior2, "prior2")


  tmp <- .ss.cons.binom.q(quantiles, accepted.pdiff, prior1, prior2,
                                            n.start, n.max, both=T)

  tmp <- c(tmp, list(quantiles=quantiles, accepted.pdiff=accepted.pdiff, prior1=prior1, prior2=prior2, n.start=n.start, n.max=n.max))
  tmp
} # end of ss.cons.binom.avg.q.bothmarg


ss.cons.binom.prob.q <- function(quantiles, accepted.pdiff, prior1, prior2, prior1.mixture.wt=0.5, prob=0.5,
                                                   n.start=1000, n.max=1e7)
{
  # prior1 and prior2 must be lists with dimensions alpha & beta
  tmp <- .ss.cons.binom.check.list(prior1, "prior1")
  tmp <- .ss.cons.binom.check.list(prior2, "prior2")


  tmp <- .ss.cons.binom.q(quantiles, accepted.pdiff, prior1, prior2,
                                            n.start, n.max, target=prob, prior1.mixture.wt=prior1.mixture.wt, return.prob=T, increasing.outcome.with.n=T)

  tmp <- c(tmp, list(quantiles=quantiles, accepted.pdiff=accepted.pdiff, prior1=prior1, prior2=prior2, prior1.mixture.wt=prior1.mixture.wt, prob=prob, n.start=n.start, n.max=n.max))
  tmp
} # end of ss.cons.binom.prob.q


ss.cons.binom.prob.q.mymarg <- function(quantiles, accepted.pdiff, prior1, prior2, clinical.prior, prob=0.5,
                                                          n.start=1000, n.max=1e7)
{
  # prior1, prior2 and clinical.prior must be lists with dimensions alpha & beta
  tmp <- .ss.cons.binom.check.list(prior1, "prior1")
  tmp <- .ss.cons.binom.check.list(prior2, "prior2")
  tmp <- .ss.cons.binom.check.list(clinical.prior, "clinical.prior")

  tmp <- .ss.cons.binom.q(quantiles, accepted.pdiff, prior1, prior2,
                                            n.start, n.max, clinical.prior=clinical.prior, 
                                            target=prob, return.prob=T, increasing.outcome.with.n=T)

  tmp <- c(tmp, list(quantiles=quantiles, accepted.pdiff=accepted.pdiff, prior1=prior1, prior2=prior2, clinical.prior=clinical.prior, prob=prob, n.start=n.start, n.max=n.max))
  tmp
} # end of ss.cons.binom.prob.q.mymarg


ss.cons.binom.prob.q.bothmarg <- function(quantiles, accepted.pdiff, prior1, prior2, prob=0.5,
                                                            n.start=1000, n.max=1e7)
{
  # prior1 and prior2 must be lists with dimensions alpha & beta
  tmp <- .ss.cons.binom.check.list(prior1, "prior1")
  tmp <- .ss.cons.binom.check.list(prior2, "prior2")


  tmp <- .ss.cons.binom.q(quantiles, accepted.pdiff, prior1, prior2,
                                            n.start, n.max, target=prob, both=T, return.prob=T, increasing.outcome.with.n=T)

  tmp <- c(tmp, list(quantiles=quantiles, accepted.pdiff=accepted.pdiff, prior1=prior1, prior2=prior2, prob=prob, n.start=n.start, n.max=n.max))
  tmp
} # end of ss.cons.binom.prob.q.bothmarg


ss.plot <- function(ss.out, zero=0.1, show.fit=T)
{
  log.n <- log(pmax(zero, ss.out$n.visited))
  n.plot <- pmax(exp(zero), ss.out$n.visited)
  out <- ss.out$outcome
  plot(n.plot, out, xlab='Sample Size', ylab='Outcome', xlog=T)
  
  if (show.fit)
  {
    df <- data.frame(out=out, log.n=log.n, log.n2 = log.n^2)
    glm.out <- glm(out~log.n+log.n2, data=df)
    log.n <- log(pmax(zero, par('usr')[c(1,2)]))
    log.n <- seq(from=log.n[1], to=log.n[2], length=300)
    df <- data.frame(log.n=log.n, log.n2=log.n^2)
    predict.out <- predict(glm.out, df)
    points(exp(log.n), predict.out, type='l', col='orange')
  }
} # end of ss.plot


# ------------------------------------------------------------------------------
# --- Definition of internal functions, not be called directly by user ---------
#     (will be called by other functions with proper arguments)        ---------


.ss.cons.binom.hpdlimits <- function(accepted.pdiff, prior1, prior2, 
                                               level, fast.approx,
                                               n.start, n.max, next.n=.ss.nextn4nonrndoutcomes,
                                               clinical.prior=list(), 
                                               target, both=F, prior1.mixture.wt=ifelse(both,-1,0), return.worst=F, return.prob=F, increasing.outcome.with.n=F, epsilon=1e-8)
{
  # prior1, prior2 and clinical.prior must be lists with dimensions alpha & beta
      
  alphas <- c(prior1$alpha, prior2$alpha, clinical.prior$alpha)
  betas  <- c(prior1$beta,  prior2$beta,  clinical.prior$beta)
  use.clinical.prior <- .ss.cons.defined.list(clinical.prior)
  
  if (!is.loaded('BinomConsensus')) 
  {
    path <- "c:/users/patrick.belisle/My Documents/Home/SampleSize/ConsensusBased/binom/dll"
    file.name <- "ConsensusBinomial"
    Running.on.64Bytes <- length(grep("64", R.Version()$arch)) > 0
    file.suffx <- character(0)
    if (!Running.on.64Bytes) file.suffx <- "-32bytes"
    dll <- paste(c(path, "/", file.name, file.suffx, ".dll"), collapse="")
    dyn.load(dll)
  }

  # Initial values

  n <- min(n.start, n.max)
  step0 <- max(ceiling(n/100), 10) # Initial step
  outcome <- numeric(0)
  n.visited <- numeric(0)

  out <- list(continue=T, target=target, increasing.outcome.with.n=increasing.outcome.with.n, step0=step0, n.max=n.max)


  while (out$continue)
  {
    z <- .C("BinomConsensus", as.double(alphas), as.double(betas), as.integer(n),
             as.double(level), as.integer(fast.approx),
             out=double(1), as.integer(return.worst), as.integer(return.prob), as.double(prior1.mixture.wt), as.integer(use.clinical.prior),
             as.double(accepted.pdiff), as.double(epsilon))

    outcome <- c(z$out, outcome)
    n.visited <- c(n, n.visited)

    out$outcome <- outcome
    out$n.visited <- n.visited

    nextn <- next.n(out)
    out$continue <- nextn$continue
    n <- nextn$n
  }

  list(n=n, n.visited=rev(out$n.visited), outcome=rev(out$outcome))
} # end of .ss.cons.binom.hpdlimits


.ss.cons.binom.cdf <- function(cdf.points, accepted.cdf.diff, 
                                          prior1, prior2,
                                          n.start, n.max, next.n=.ss.nextn4nonrndoutcomes, 
                                          clinical.prior=list(),
                                          target=accepted.cdf.diff, both=F, prior1.mixture.wt=ifelse(both,-1,0), return.worst=F, return.prob=F, increasing.outcome.with.n=F)
{
  # prior1, prior2 and clinical.prior must be lists with dimensions alpha & beta
    
  alpha1 <- prior1$alpha
  beta1  <- prior1$beta
  alpha2 <- prior2$alpha
  beta2  <- prior2$beta
  alpha3 <- clinical.prior$alpha
  beta3  <- clinical.prior$beta

  # Initial values

  n <- min(n.start, n.max)
  step0 <- max(ceiling(n/100), 10) # Initial step
  outcome <- numeric(0)
  n.visited <- numeric(0)

  out <- list(continue=T, target=target, increasing.outcome.with.n=increasing.outcome.with.n, step0=step0, n.max=n.max)


  ss.cons.binom.cdf.out <- function(n, alpha1, beta1, alpha2, beta2, alpha3, beta3, cdf.points, accepted.cdf.diff, return.worst, return.prob, prior1.mixture.wt)
  {
    use.clinical.prior <- length(alpha3) > 0
    
    dBinomialBeta <- function(alpha, beta, n)
    {
      x <- 0:n
      exp(lgamma(alpha+beta)-lgamma(alpha)-lgamma(beta)-lgamma(alpha+beta+n)
         +lgamma(n+1)-lgamma(x+1)-lgamma(n-x+1)+lgamma(alpha+x)+lgamma(beta+n-x))
    }

    # Compute marginal distribution

    if (!return.worst)
    {
      if (use.clinical.prior)
      {
        mx <- dBinomialBeta(alpha3, beta3, n)
      }
      else
      {
        m1 <- dBinomialBeta(alpha1, beta1, n)
        m2 <- dBinomialBeta(alpha2, beta2, n)
        if (prior1.mixture.wt < 0)
        {
          mx <- c(m1, m2)
        }
        else
        {
          mx <- prior1.mixture.wt*m1 + (1-prior1.mixture.wt)*m2
        }
      }
      mx <- matrix(mx, nrow=n+1)
    }

    # Compute absolute differences g_n(x) in cumulative posterior probabilites

    x <- 0:n

    lq <- length(cdf.points)
    rep.cdf.points <- rep(cdf.points, rep(n+1,lq))

    p1 <- pbeta(rep.cdf.points, alpha1+x, beta1+n-x)
    p2 <- pbeta(rep.cdf.points, alpha2+x, beta2+n-x)
    pdiff <- matrix(abs(p2-p1), ncol=lq)

    if (return.worst)
    {
      out <- max(pdiff)
    }
    else 
    {
      gn.x <- apply(pdiff,1,max)
      
      if (return.prob)
      {
        out <- gn.x <= accepted.cdf.diff 
      }
      else
      {
        out <- gn.x 
      }
      
      out <- t(mx)%*%matrix(out,ncol=1)
      out <- ifelse(return.prob, min(out), max(out))
    }
    
    out
  } # end of ss.cons.binom.cdf.out

  # ---

  while (out$continue)
  {
    z <- ss.cons.binom.cdf.out(n, alpha1, beta1, alpha2, beta2, alpha3, beta3, cdf.points, accepted.cdf.diff, return.worst, return.prob, prior1.mixture.wt)

    outcome <- c(z, outcome)
    n.visited <- c(n, n.visited)

    out$outcome <- outcome
    out$n.visited <- n.visited

    nextn <- next.n(out)
    out$continue <- nextn$continue
    n <- nextn$n
  }

  list(n=n, n.visited=rev(out$n.visited), outcome=rev(out$outcome))
} # end of .ss.cons.binom.cdf


.ss.cons.binom.q <- function(quantiles, accepted.pdiff,
                                               prior1, prior2,
                                               n.start, n.max, next.n=.ss.nextn4nonrndoutcomes, 
                                               clinical.prior=list(),
                                               target=accepted.pdiff, both=F, prior1.mixture.wt=ifelse(both,-1,0), return.worst=F, return.prob=F, increasing.outcome.with.n=F)
{
  # prior1, prior2 and clinical.prior must be lists with dimensions alpha & beta

  alpha1 <- prior1$alpha
  beta1  <- prior1$beta
  alpha2 <- prior2$alpha
  beta2  <- prior2$beta
  alpha3 <- clinical.prior$alpha
  beta3  <- clinical.prior$beta

  # Initial values

  n <- min(n.start, n.max)
  step0 <- max(ceiling(n/100), 10) # Initial step
  outcome <- numeric(0)
  n.visited <- numeric(0)

  out <- list(continue=T, target=target, increasing.outcome.with.n=increasing.outcome.with.n, step0=step0, n.max=n.max)


  ss.cons.binom.q.out <- function(n, alpha1, beta1, alpha2, beta2, alpha3, beta3, quantiles, accepted.pdiff, return.worst, return.prob, prior1.mixture.wt)
  {
    use.clinical.prior <- length(alpha3) > 0
    
    dBinomialBeta <- function(alpha, beta, n)
    {
      x <- 0:n
      exp(lgamma(alpha+beta)-lgamma(alpha)-lgamma(beta)-lgamma(alpha+beta+n)
         +lgamma(n+1)-lgamma(x+1)-lgamma(n-x+1)+lgamma(alpha+x)+lgamma(beta+n-x))
    }

    # Compute marginal distribution

    if (!return.worst)
    {
      if (use.clinical.prior)
      {
        mx <- dBinomialBeta(alpha3, beta3, n)
      }
      else
      {
        m1 <- dBinomialBeta(alpha1, beta1, n)
        m2 <- dBinomialBeta(alpha2, beta2, n)
        if (prior1.mixture.wt < 0)
        {
          mx <- c(m1, m2)
        }
        else
        {
          mx <- prior1.mixture.wt*m1 + (1-prior1.mixture.wt)*m2
        }
      }
      mx <- matrix(mx, nrow=n+1)
    }

    # Compute absolute differences g_n(x) in cumulative posterior probabilites

    x <- 0:n

    q.len <- length(quantiles)
    rep.quantiles <- rep(quantiles, rep(n+1,q.len))

    q1 <- qbeta(rep.quantiles, alpha1+x, beta1+n-x)
    q2 <- qbeta(rep.quantiles, alpha2+x, beta2+n-x)
    qdiff <- matrix(abs(q2-q1), ncol=q.len)

    if (return.worst)
    {
      out <- max(qdiff)
    }
    else
    {
      gn.x <- apply(qdiff,1,max)

      if (return.prob)
      {
        out <- gn.x <= accepted.pdiff
      }
      else
      {
        out <- gn.x
      }

      out <- t(mx)%*%matrix(out,ncol=1)
      out <- ifelse(return.prob, min(out), max(out))
    }

    out
  } # end of ss.cons.binom.q.out

  # ---

  while (out$continue)
  {
    z <- ss.cons.binom.q.out(n, alpha1, beta1, alpha2, beta2, alpha3, beta3, quantiles, accepted.pdiff, return.worst, return.prob, prior1.mixture.wt)

    outcome <- c(z, outcome)
    n.visited <- c(n, n.visited)

    out$outcome <- outcome
    out$n.visited <- n.visited

    nextn <- next.n(out)
    out$continue <- nextn$continue
    n <- nextn$n
  }

  list(n=n, n.visited=rev(out$n.visited), outcome=rev(out$outcome))
} # end of .ss.cons.binom.q


.ss.nextn4nonrndoutcomes <- function(mylist)
{
  # Function used to perform the bisectional search

  ##################################################################
  # mylist: consists in a list of the following arguments:
  # -------
  #
  # n.visited
  # outcome
  # target
  # step0
  # increasing.outcome.with.n
  # n.max
  #
  ##################################################################

  continue <- T

  sufficient.n <- (mylist$increasing.outcome.with.n & mylist$outcome >= mylist$target) | (!mylist$increasing.outcome.with.n & mylist$outcome <= mylist$target)
  n.visited <- mylist$n.visited

  if (all(sufficient.n))
  {
    n.visited <- sort(n.visited)

    if (n.visited[1] == 0)
    {
      n <- 0
      continue <- F
    }
    else
    {
      if (length(n.visited) == 1)
      {
        n <- max(n.visited - mylist$step0, 0)
      }
      else
      {
        n <- max(n.visited[1] - 2*diff(n.visited[1:2]), 0)
      }
    }
  }
  else if (all(!sufficient.n))
  {
    n.visited <- rev(sort(n.visited))

    if (n.visited[1] == mylist$n.max)
    {
      n <- Inf
      continue <- F
    }
    else
    {
      if (length(n.visited) == 1)
      {
        n <- n.visited + mylist$step0
      }
      else
      {
        n <- n.visited[1] + 2 * (n.visited[1]-n.visited[2])
      }

      n.visited <- min(n.visited, mylist$n.max)
    }
  }
  else
  {
    smallest.sufficient.n  <- min(n.visited[sufficient.n])
    largest.insufficient.n <- max(n.visited[!sufficient.n])

    if ((smallest.sufficient.n-largest.insufficient.n) == 1)
    {
      n <- smallest.sufficient.n
      continue <- F
    }
    else
    {
      n <- floor((smallest.sufficient.n + largest.insufficient.n)/2)
    }
  }
  
  if (n > mylist$n.max)
  {
    n <- Inf
    continue <- F
  }

  # If continue=F, then n indicates the optimal sample size found

  list(n=n, continue=continue)
} # end of .ss.nextn4nonrndoutcomes


.ss.cons.binom.check.list <- function(test.prior=list(), prior.name)
{
  if (missing(test.prior))
  {
    stop(paste(c(prior.name, " list undefined."), collapse=''), call.=F) 
  }
  else
  {
    tmp <- try(paste(test.prior), silent=T)
    if (class(tmp) != "try-error")
    {
      dim.names <- names(test.prior)
      present.dim <- match(c("alpha", "beta"), dim.names)
      if (any(is.na(present.dim))) stop(paste(c("Both alpha and beta must be dimensions of ", prior.name, " list."), collapse=''), call.=F)
    }
    else
    {
      stop(paste(c(prior.name, " list undefined."), collapse=''), call.=F)
    }
  }
} # end of .ss.cons.binom.check.list


.ss.cons.defined.list <- function(mylist){ifelse(length(mylist) == 0, F, any(unlist(lapply(mylist, length)) > 0))}



