# Consensus-Based Sample Size - Difference between Two Normal Means
#   Version 0.11 (November 2014)

                                                                                                                   
ss.cons.2norm.avg.hpdlimits <- function(accepted.diff, prior1, prior2, prior1.mixture.wt=0.5,
                                                                         level=0.95, n2.n1.ratio=1, n.start=1000, n.max=100000, sim.size=50000, max.cons.steps.same.dir=3)
{
  # prior1 and prior2 must be lists with dimensions mu0, n0, prec.shape and prec.rate
  tmp <- .ss.cons.2norm.check.list(prior1, "prior1")
  tmp <- .ss.cons.2norm.check.list(prior2, "prior2")
  
  tmp <- .ss.cons.2norm.known.sds(prior1, prior2)
  
  
  if (all(tmp$known))
  {
    out <- .ss.cons.2norm.hpdlimits.known.sds(tmp, accepted.diff, level=level, prior1.mixture.wt=prior1.mixture.wt, n2.n1.ratio=n2.n1.ratio,
              n.start=n.start, n.max=n.max)
  
    out <- c(out, list(accepted.diff=accepted.diff, prior1=prior1, prior2=prior2,
                       prior1.mixture.wt=prior1.mixture.wt, level=level, n2.n1.ratio=n2.n1.ratio, n.start=n.start, n.max=n.max,
                       exact.results=T))
  }
  else
  {
    out <- .ss.cons.2norm.hpdlimits(accepted.diff, prior1, prior2,
                                                                     level, sim.size, n.start, n.max, prior1.mixture.wt=prior1.mixture.wt,
                                                                     target=accepted.diff, n2.n1.ratio=n2.n1.ratio, mcs=max.cons.steps.same.dir)

    out <- c(out, list(accepted.diff=accepted.diff, prior1=prior1, prior2=prior2, prior1.mixture.wt=prior1.mixture.wt, 
                       level=level, n2.n1.ratio=n2.n1.ratio, n.start=n.start, n.max=n.max, sim.size=sim.size,
                       max.cons.steps.same.dir=max.cons.steps.same.dir, exact.results=F))
  }
  
  return(out)
} # end of ss.cons.2norm.avg.hpdlimits


ss.cons.2norm.avg.hpdlimits.mymarg <- function(accepted.diff, prior1, prior2, clinical.prior,
                                                                                level=0.95, n2.n1.ratio=1, n.start=1000, n.max=100000, sim.size=50000,
                                                                                max.cons.steps.same.dir=3)
{
  # prior1 and prior2 must be lists with dimensions mu0, n0, prec.shape and prec.rate
  tmp <- .ss.cons.2norm.check.list(prior1, "prior1")
  tmp <- .ss.cons.2norm.check.list(prior2, "prior2")
  tmp <- .ss.cons.2norm.check.list(clinical.prior, "clinical.prior")

  tmp <- .ss.cons.2norm.known.sds(prior1, prior2)
  
  
  if (all(tmp$known))
  {
    out <- .ss.cons.2norm.hpdlimits.known.sds(tmp, accepted.diff, level=level, clinical.prior=clinical.prior, n2.n1.ratio=n2.n1.ratio,
              n.start=n.start, n.max=n.max)
  
    out <- c(out, list(accepted.diff=accepted.diff, prior1=prior1, prior2=prior2, clinical.prior=clinical.prior, 
                       level=level, n2.n1.ratio=n2.n1.ratio, n.start=n.start, n.max=n.max, exact.results=T))
  }
  else
  {
    out <- .ss.cons.2norm.hpdlimits(accepted.diff, prior1, prior2,
                                                                     level, sim.size, n.start, n.max,
                                                                     clinical.prior=clinical.prior, target=accepted.diff, n2.n1.ratio=n2.n1.ratio, mcs=max.cons.steps.same.dir)


    out <- c(out, list(accepted.diff=accepted.diff, prior1=prior1, prior2=prior2, clinical.prior=clinical.prior,
                       level=level, n2.n1.ratio=n2.n1.ratio, n.start=n.start, n.max=n.max, sim.size=sim.size,
                       max.cons.steps.same.dir=max.cons.steps.same.dir, exact.results=F))
  }
  
  return(out)
} # end of ss.cons.2norm.avg.hpdlimits.mymarg


ss.cons.2norm.avg.hpdlimits.bothmarg <- function(accepted.diff, prior1, prior2,
                                                 level=0.95, n2.n1.ratio=1, n.start=1000, n.max=100000, sim.size=50000,
                                                 max.cons.steps.same.dir=3)
{
  # prior1 and prior2 must be lists with dimensions mu0, n0, prec.shape and prec.rate
  tmp <- .ss.cons.2norm.check.list(prior1, "prior1")
  tmp <- .ss.cons.2norm.check.list(prior2, "prior2")

  tmp <- .ss.cons.2norm.known.sds(prior1, prior2)
  
  
  if (all(tmp$known))
  {
    out <- .ss.cons.2norm.hpdlimits.known.sds(tmp, accepted.diff, level=level, both=T, n2.n1.ratio=n2.n1.ratio,
              n.start=n.start, n.max=n.max)
  
    out <- c(out, list(accepted.diff=accepted.diff, prior1=prior1, prior2=prior2,
                       level=level, n2.n1.ratio=n2.n1.ratio, n.start=n.start, n.max=n.max, exact.results=T))
  }
  else
  {
    out <- .ss.cons.2norm.hpdlimits(accepted.diff, prior1, prior2,
                                                                     level, sim.size, n.start, n.max,
                                                                     both=T, target=accepted.diff, n2.n1.ratio=n2.n1.ratio, mcs=max.cons.steps.same.dir)

    out <- c(out, list(accepted.diff=accepted.diff, prior1=prior1, prior2=prior2,
                       level=level, n2.n1.ratio=n2.n1.ratio, n.start=n.start, n.max=n.max, sim.size=sim.size,
                       max.cons.steps.same.dir=max.cons.steps.same.dir, exact.results=F))
  }
  
  return(out)
} # end of ss.cons.2norm.avg.hpdlimits.bothmarg


ss.cons.2norm.prob.hpdlimits <- function(accepted.diff, prior1, prior2, prior1.mixture.wt=0.5,
                                                                          prob=0.5, level=0.95, n2.n1.ratio=1, n.start=1000, n.max=100000, sim.size=50000,
                                                                          max.cons.steps.same.dir=3)
{
  # prior1 and prior2 must be lists with dimensions mu0, n0, prec.shape and prec.rate
  tmp <- .ss.cons.2norm.check.list(prior1, "prior1")
  tmp <- .ss.cons.2norm.check.list(prior2, "prior2")
  
  tmp <- .ss.cons.2norm.known.sds(prior1, prior2)
  

  if (all(tmp$known))
  {
    out <- .ss.cons.2norm.hpdlimits.known.sds(tmp, accepted.diff, level=level, prior1.mixture.wt=prior1.mixture.wt, n2.n1.ratio=n2.n1.ratio,
              n.start=n.start, n.max=n.max, target=prob, compute.prob=T)
  
    out <- c(out, list(accepted.diff=accepted.diff,  
                       prior1=prior1, prior2=prior2, prior1.mixture.wt=prior1.mixture.wt, prob=prob,
                       level=level, n2.n1.ratio=n2.n1.ratio, n.start=n.start, n.max=n.max, exact.results=T))
  }
  else
  {
    out <- .ss.cons.2norm.hpdlimits(accepted.diff, prior1, prior2,
                                                                     level, sim.size, n.start, n.max, prior1.mixture.wt=prior1.mixture.wt,
                                                                     return.prob=T, target=prob, n2.n1.ratio=n2.n1.ratio, mcs=max.cons.steps.same.dir)

    out <- c(out, list(accepted.diff=accepted.diff, 
                       prior1=prior1, prior2=prior2, prior1.mixture.wt=prior1.mixture.wt, prob=prob, 
                       level=level, n2.n1.ratio=n2.n1.ratio, n.start=n.start, n.max=n.max, 
                       sim.size=sim.size, max.cons.steps.same.dir=max.cons.steps.same.dir, exact.results=F))
  }
  
  return(out)
} # end of ss.cons.2norm.prob.hpdlimits


ss.cons.2norm.prob.hpdlimits.mymarg <- function(accepted.diff, prior1, prior2, clinical.prior,
                                                 prob=0.5, level=0.95, n2.n1.ratio=1, n.start=1000, n.max=100000, sim.size=50000,
                                                 max.cons.steps.same.dir=3)
{
  # prior1 and prior2 must be lists with dimensions mu0, n0, prec.shape and prec.rate
  tmp <- .ss.cons.2norm.check.list(prior1, "prior1")
  tmp <- .ss.cons.2norm.check.list(prior2, "prior2")
  tmp <- .ss.cons.2norm.check.list(clinical.prior, "clinical.prior")

  tmp <- .ss.cons.2norm.known.sds(prior1, prior2)
  
  
  if (all(tmp$known))
  {
    out <- .ss.cons.2norm.hpdlimits.known.sds(tmp, accepted.diff, level=level, clinical.prior=clinical.prior, n2.n1.ratio=n2.n1.ratio,
              n.start=n.start, n.max=n.max, target=prob, compute.prob=T)
  
    out <- c(out, list(accepted.diff=accepted.diff, prior1=prior1, prior2=prior2, clinical.prior=clinical.prior, prob=prob, 
                       level=level, n2.n1.ratio=n2.n1.ratio, n.start=n.start, n.max=n.max, exact.results=T))
  }
  else
  {    
    out <- .ss.cons.2norm.hpdlimits(accepted.diff, prior1, prior2,
                                                                     level, sim.size, n.start, n.max,
                                                                     clinical.prior=clinical.prior, return.prob=T, target=prob, n2.n1.ratio=n2.n1.ratio, mcs=max.cons.steps.same.dir)

    out <- c(out, list(accepted.diff=accepted.diff, prior1=prior1, prior2=prior2, clinical.prior=clinical.prior, prob=prob,
                       level=level, n2.n1.ratio=n2.n1.ratio, n.start=n.start, n.max=n.max, 
                       sim.size=sim.size, max.cons.steps.same.dir=max.cons.steps.same.dir, exact.results=F))
  }
  
  return(out)
} # end of ss.cons.2norm.prob.hpdlimits.mymarg


ss.cons.2norm.prob.hpdlimits.bothmarg <- function(accepted.diff, prior1, prior2, prob=0.5,
                                                 level=0.95, n2.n1.ratio=1, n.start=1000, n.max=100000, sim.size=50000,
                                                 max.cons.steps.same.dir=3)
{
  # prior1 and prior2 must be lists with dimensions mu0, n0, prec.shape and prec.rate
  tmp <- .ss.cons.2norm.check.list(prior1, "prior1")
  tmp <- .ss.cons.2norm.check.list(prior2, "prior2")

  tmp <- .ss.cons.2norm.known.sds(prior1, prior2)

  if (all(tmp$known))
  {
    out <- .ss.cons.2norm.hpdlimits.known.sds(tmp, accepted.diff, level=level, both=T, n2.n1.ratio=n2.n1.ratio,
              n.start=n.start, n.max=n.max, target=prob, compute.prob=T)
  
    out <- c(out, list(accepted.diff=accepted.diff, prior1=prior1, prior2=prior2, prob=prob,
                       level=level, n2.n1.ratio=n2.n1.ratio, n.start=n.start, n.max=n.max, exact.results=T))
  }
  else
  {
    out <- .ss.cons.2norm.hpdlimits(accepted.diff, prior1, prior2,
                                                                     level, sim.size, n.start, n.max,
                                                                     return.prob=T, both=T, target=prob, n2.n1.ratio=n2.n1.ratio, mcs=max.cons.steps.same.dir)

    out <- c(out, list(accepted.diff=accepted.diff, prior1=prior1, prior2=prior2, prob=prob,
                       level=level, n2.n1.ratio=n2.n1.ratio, n.start=n.start, n.max=n.max, sim.size=sim.size,
                       max.cons.steps.same.dir=max.cons.steps.same.dir, exact.results=F))
  }
  
  return(out)
} # end of ss.cons.2norm.prob.hpdlimits.bothmarg


ss.cons.2norm.avg.cdf <- function(cdf.points, accepted.cdf.diff, prior1, prior2, prior1.mixture.wt=0.5, 
                                                 n2.n1.ratio=1, n.start=1000, n.max=100000, sim.size=50000,
                                                 max.cons.steps.same.dir=3)
{
  # prior1 and prior2 must be lists with dimensions mu0, n0, prec.shape and prec.rate
  tmp <- .ss.cons.2norm.check.list(prior1, "prior1")
  tmp <- .ss.cons.2norm.check.list(prior2, "prior2")

  tmp <- .ss.cons.2norm.known.sds(prior1, prior2)
  
  if (all(tmp$known) & length(cdf.points) == 1)
  {
    out <- .ss.cons.2norm.cdf.known.sds(tmp, cdf.points, accepted.cdf.diff, prior1.mixture.wt=prior1.mixture.wt, n2.n1.ratio=n2.n1.ratio,
              n.start=n.start, n.max=n.max)
  
    out <- c(out, list(cdf.points=cdf.points, accepted.cdf.diff=accepted.cdf.diff, 
                       prior1=prior1, prior2=prior2, prior1.mixture.wt=prior1.mixture.wt, 
                       n2.n1.ratio=n2.n1.ratio, n.start=n.start, n.max=n.max, exact.results=T))
  }
  else
  {
    out <- .ss.cons.2norm.cdf(cdf.points, accepted.cdf.diff, prior1, prior2,
                                                                sim.size, n.start, n.max, prior1.mixture.wt=prior1.mixture.wt, n2.n1.ratio=n2.n1.ratio,
                                                                mcs=max.cons.steps.same.dir)

    out <- c(out, list(cdf.points=cdf.points, accepted.cdf.diff=accepted.cdf.diff, prior1=prior1, prior2=prior2,
                       prior1.mixture.wt=prior1.mixture.wt, n2.n1.ratio=n2.n1.ratio, n.start=n.start, n.max=n.max, sim.size=sim.size,
                       max.cons.steps.same.dir=max.cons.steps.same.dir, exact.results=F))
  }
  
  return(out)
} # end of ss.cons.2norm.avg.cdf


ss.cons.2norm.avg.cdf.mymarg <- function(cdf.points, accepted.cdf.diff, prior1, prior2, clinical.prior,
                                                 n2.n1.ratio=1, n.start=1000, n.max=100000, sim.size=50000,
                                                 max.cons.steps.same.dir=3)
{
  # prior1 and prior2 must be lists with dimensions mu0, n0, prec.shape and prec.rate
  tmp <- .ss.cons.2norm.check.list(prior1, "prior1")
  tmp <- .ss.cons.2norm.check.list(prior2, "prior2")
  tmp <- .ss.cons.2norm.check.list(clinical.prior, "clinical.prior")

  tmp <- .ss.cons.2norm.known.sds(prior1, prior2)
  
  
  if (all(tmp$known) & length(cdf.points) == 1)
  {
    out <- .ss.cons.2norm.cdf.known.sds(tmp, cdf.points, accepted.cdf.diff, clinical.prior=clinical.prior, n2.n1.ratio=n2.n1.ratio,
              n.start=n.start, n.max=n.max)
  
    out <- c(out, list(cdf.points=cdf.points, accepted.cdf.diff=accepted.cdf.diff, 
                       prior1=prior1, prior2=prior2, clinical.prior=clinical.prior, 
                       n2.n1.ratio=n2.n1.ratio, n.start=n.start, n.max=n.max, exact.results=T))
  }
  else
  {
    out <- .ss.cons.2norm.cdf(cdf.points, accepted.cdf.diff, prior1, prior2,
                                                                sim.size, n.start, n.max, clinical.prior=clinical.prior, n2.n1.ratio=n2.n1.ratio,
                                                                mcs=max.cons.steps.same.dir)

    out <- c(out, list(cdf.points=cdf.points, accepted.cdf.diff=accepted.cdf.diff,
                       prior1=prior1, prior2=prior2, clinical.prior=clinical.prior,
                       n2.n1.ratio=n2.n1.ratio, n.start=n.start, n.max=n.max, sim.size=sim.size,
                       max.cons.steps.same.dir=max.cons.steps.same.dir, exact.results=F))
  }
  
  return(out)
} # end of ss.cons.2norm.avg.cdf.mymarg


ss.cons.2norm.avg.cdf.bothmarg <- function(cdf.points, accepted.cdf.diff, prior1, prior2,
                                                 n2.n1.ratio=1, n.start=1000, n.max=100000, sim.size=50000,
                                                 max.cons.steps.same.dir=3)
{
  # prior1 and prior2 must be lists with dimensions mu0, n0, prec.shape and prec.rate
  tmp <- .ss.cons.2norm.check.list(prior1, "prior1")
  tmp <- .ss.cons.2norm.check.list(prior2, "prior2")

  tmp <- .ss.cons.2norm.known.sds(prior1, prior2)
  
  if (all(tmp$known) & length(cdf.points) == 1)
  {
    out <- .ss.cons.2norm.cdf.known.sds(tmp, cdf.points, accepted.cdf.diff, both=T, n2.n1.ratio=n2.n1.ratio,
              n.start=n.start, n.max=n.max)
  
    out <- c(out, list(cdf.points=cdf.points, accepted.cdf.diff=accepted.cdf.diff, 
                       prior1=prior1, prior2=prior2,
                       n2.n1.ratio=n2.n1.ratio, n.start=n.start, n.max=n.max, exact.results=T))
  }
  else
  {
    out <- .ss.cons.2norm.cdf(cdf.points, accepted.cdf.diff, prior1, prior2,
                                                                sim.size, n.start, n.max, both=T, n2.n1.ratio=n2.n1.ratio, mcs=max.cons.steps.same.dir)

    out <- c(out, list(cdf.points=cdf.points, accepted.cdf.diff=accepted.cdf.diff, 
                       prior1=prior1, prior2=prior2,
                       n2.n1.ratio=n2.n1.ratio, n.start=n.start, n.max=n.max, sim.size=sim.size,
                       max.cons.steps.same.dir=max.cons.steps.same.dir, exact.results=F))
  }
  
  return(out)
} # end of ss.cons.2norm.avg.cdf.bothmarg


ss.cons.2norm.prob.cdf <- function(cdf.points, accepted.cdf.diff, prior1, prior2, prior1.mixture.wt=0.5, 
                                                 prob=0.5, n2.n1.ratio=1, n.start=1000, n.max=100000, sim.size=50000,
                                                 max.cons.steps.same.dir=3)
{
  # prior1 and prior2 must be lists with dimensions mu0, n0, prec.shape and prec.rate
  tmp <- .ss.cons.2norm.check.list(prior1, "prior1")
  tmp <- .ss.cons.2norm.check.list(prior2, "prior2")
  

  out <- .ss.cons.2norm.cdf(cdf.points, accepted.cdf.diff, prior1, prior2,
                                                              sim.size, n.start, n.max, prior1.mixture.wt=prior1.mixture.wt,
                                                              return.prob=T, target=prob, n2.n1.ratio=n2.n1.ratio, mcs=max.cons.steps.same.dir)

  out <- c(out, list(cdf.points=cdf.points, accepted.cdf.diff=accepted.cdf.diff, 
                     prior1=prior1, prior2=prior2, prior1.mixture.wt=prior1.mixture.wt, prob=prob, 
                     n2.n1.ratio=n2.n1.ratio, n.start=n.start, n.max=n.max, sim.size=sim.size,
                     max.cons.steps.same.dir=max.cons.steps.same.dir, exact.results=F))
  
  return(out)
} # end of ss.cons.2norm.prob.cdf


ss.cons.2norm.prob.cdf.mymarg <- function(cdf.points, accepted.cdf.diff, prior1, prior2, clinical.prior,
                                                 prob=0.5, n2.n1.ratio=1, n.start=1000, n.max=100000, sim.size=50000,
                                                 max.cons.steps.same.dir=3)
{
  # prior1 and prior2 must be lists with dimensions mu0, n0, prec.shape and prec.rate
  tmp <- .ss.cons.2norm.check.list(prior1, "prior1")
  tmp <- .ss.cons.2norm.check.list(prior2, "prior2")
  tmp <- .ss.cons.2norm.check.list(clinical.prior, "clinical.prior")

  
  out <- .ss.cons.2norm.cdf(cdf.points, accepted.cdf.diff, prior1, prior2,
                                                              sim.size, n.start, n.max,
                                                              clinical.prior=clinical.prior, return.prob=T, target=prob, n2.n1.ratio=n2.n1.ratio,
                                                              mcs=max.cons.steps.same.dir)

  out <- c(out, list(cdf.points=cdf.points, accepted.cdf.diff=accepted.cdf.diff,
                     prior1=prior1, prior2=prior2, clinical.prior=clinical.prior, prob=prob,
                     n2.n1.ratio=n2.n1.ratio, n.start=n.start, n.max=n.max, sim.size=sim.size,
                     max.cons.steps.same.dir=max.cons.steps.same.dir, exact.results=F))
  
  return(out)
} # end of ss.cons.2norm.prob.cdf.mymarg


ss.cons.2norm.prob.cdf.bothmarg <- function(cdf.points, accepted.cdf.diff, prior1, prior2,
                                                 prob=0.5, n2.n1.ratio=1, n.start=1000, n.max=100000, sim.size=50000,
                                                 max.cons.steps.same.dir=3)
{
  # prior1 and prior2 must be lists with dimensions mu0, n0, prec.shape and prec.rate
  tmp <- .ss.cons.2norm.check.list(prior1, "prior1")
  tmp <- .ss.cons.2norm.check.list(prior2, "prior2")
  

  out <- .ss.cons.2norm.cdf(cdf.points, accepted.cdf.diff, prior1, prior2,
                                                              sim.size, n.start, n.max,
                                                              return.prob=T, both=T, target=prob, n2.n1.ratio=n2.n1.ratio,
                                                              mcs=max.cons.steps.same.dir)

  out <- c(out, list(cdf.points=cdf.points, accepted.cdf.diff=accepted.cdf.diff,
                     prior1=prior1, prior2=prior2, prob=prob,
                     n2.n1.ratio=n2.n1.ratio, n.start=n.start, n.max=n.max, sim.size=sim.size,
                     max.cons.steps.same.dir=max.cons.steps.same.dir, exact.results=F))
  
  return(out)
} # end of ss.cons.2norm.prob.cdf.bothmarg


ss.cons.2norm.avg.q <- function(quantiles, accepted.diff, prior1, prior2, prior1.mixture.wt=0.5, 
                                                 n2.n1.ratio=1, n.start=1000, n.max=100000, sim.size=50000,
                                                 max.cons.steps.same.dir=3)
{
  # prior1 and prior2 must be lists with dimensions mu0, n0, prec.shape and prec.rate
  tmp <- .ss.cons.2norm.check.list(prior1, "prior1")
  tmp <- .ss.cons.2norm.check.list(prior2, "prior2")

  tmp <- .ss.cons.2norm.known.sds(prior1, prior2)
  
  
  if (all(tmp$known))
  {
    out <- .ss.cons.2norm.q.known.sds(tmp, accepted.diff, quantiles, prior1.mixture.wt=prior1.mixture.wt, n2.n1.ratio=n2.n1.ratio,
              n.start=n.start, n.max=n.max)
  
    out <- c(out, list(quantiles=quantiles, accepted.diff=accepted.diff, prior1=prior1, prior2=prior2, prior1.mixture.wt=prior1.mixture.wt, 
                       n2.n1.ratio=n2.n1.ratio, n.start=n.start, n.max=n.max, exact.results=T))
  }
  else
  {
    out <- .ss.cons.2norm.q(quantiles, accepted.diff, prior1, prior2,
                                                                     sim.size, n.start, n.max, prior1.mixture.wt=prior1.mixture.wt,
                                                                     target=accepted.diff, n2.n1.ratio=n2.n1.ratio, mcs=max.cons.steps.same.dir)

    out <- c(out, list(quantiles=quantiles, accepted.diff=accepted.diff, prior1=prior1, prior2=prior2, prior1.mixture.wt=prior1.mixture.wt, 
                       n2.n1.ratio=n2.n1.ratio, n.start=n.start, n.max=n.max, sim.size=sim.size,
                       max.cons.steps.same.dir=max.cons.steps.same.dir, exact.results=F))
  }
  
  return(out)
} # end of ss.cons.2norm.avg.q


ss.cons.2norm.avg.q.mymarg <- function(quantiles, accepted.diff, prior1, prior2, clinical.prior,
                                                 n2.n1.ratio=1, n.start=1000, n.max=100000,
                                                 sim.size=50000, max.cons.steps.same.dir=3)
{
  # prior1 and prior2 must be lists with dimensions mu0, n0, prec.shape and prec.rate
  tmp <- .ss.cons.2norm.check.list(prior1, "prior1")
  tmp <- .ss.cons.2norm.check.list(prior2, "prior2")

  tmp <- .ss.cons.2norm.known.sds(prior1, prior2)
  
  if (all(tmp$known))
  {
    out <- .ss.cons.2norm.q.known.sds(tmp, accepted.diff, quantiles, clinical.prior=clinical.prior, n2.n1.ratio=n2.n1.ratio,
              n.start=n.start, n.max=n.max)
  
    out <- c(out, list(quantiles=quantiles, accepted.diff=accepted.diff, prior1=prior1, prior2=prior2, clinical.prior=clinical.prior, 
                       n2.n1.ratio=n2.n1.ratio, n.start=n.start, n.max=n.max, exact.results=T))
  }
  else
  {
    out <- .ss.cons.2norm.q(quantiles, accepted.diff, prior1, prior2,
                                                                     sim.size, n.start, n.max,
                                                                     clinical.prior=clinical.prior, target=accepted.diff, n2.n1.ratio=n2.n1.ratio, mcs=max.cons.steps.same.dir)

    out <- c(out, list(quantiles=quantiles, accepted.diff=accepted.diff, prior1=prior1, prior2=prior2, clinical.prior=clinical.prior,
                       n2.n1.ratio=n2.n1.ratio, n.start=n.start, n.max=n.max, sim.size=sim.size,
                       max.cons.steps.same.dir=max.cons.steps.same.dir, exact.results=F))
  }
  
  return(out)
} # end of ss.cons.2norm.avg.q.mymarg


ss.cons.2norm.avg.q.bothmarg <- function(quantiles, accepted.diff, prior1, prior2,
                                                 n2.n1.ratio=1, n.start=1000, n.max=100000, sim.size=50000,
                                                 max.cons.steps.same.dir=3)
{
  # prior1 and prior2 must be lists with dimensions mu0, n0, prec.shape and prec.rate
  tmp <- .ss.cons.2norm.check.list(prior1, "prior1")
  tmp <- .ss.cons.2norm.check.list(prior2, "prior2")

  tmp <- .ss.cons.2norm.known.sds(prior1, prior2)
  
  if (all(tmp$known))
  {
    out <- .ss.cons.2norm.q.known.sds(tmp, accepted.diff, quantiles, n2.n1.ratio=n2.n1.ratio,
              n.start=n.start, n.max=n.max)
  
    out <- c(out, list(quantiles=quantiles, accepted.diff=accepted.diff, prior1=prior1, prior2=prior2,
                       n2.n1.ratio=n2.n1.ratio, n.start=n.start, n.max=n.max, exact.results=T))
  }
  else
  {
    out <- .ss.cons.2norm.q(quantiles, accepted.diff, prior1, prior2,
                                                                     sim.size, n.start, n.max,
                                                                     both=T, target=accepted.diff, n2.n1.ratio=n2.n1.ratio, mcs=max.cons.steps.same.dir)

    out <- c(out, list(quantiles=quantiles, accepted.diff=accepted.diff, prior1=prior1, prior2=prior2,
                       n2.n1.ratio=n2.n1.ratio, n.start=n.start, n.max=n.max, 
                       sim.size=sim.size, max.cons.steps.same.dir=max.cons.steps.same.dir, exact.results=F))
  }
  
  return(out)
} # end of ss.cons.2norm.avg.q.bothmarg


ss.cons.2norm.prob.q <- function(quantiles, accepted.diff, prior1, prior2, prior1.mixture.wt=0.5, 
                                                 prob=0.5, n2.n1.ratio=1, n.start=1000, n.max=100000, sim.size=50000,
                                                 max.cons.steps.same.dir=3)
{
  # prior1 and prior2 must be lists with dimensions mu0, n0, prec.shape and prec.rate
  tmp <- .ss.cons.2norm.check.list(prior1, "prior1")
  tmp <- .ss.cons.2norm.check.list(prior2, "prior2")

  tmp <- .ss.cons.2norm.known.sds(prior1, prior2)
  
  if (all(tmp$known))
  {
    out <- .ss.cons.2norm.q.known.sds(tmp, accepted.diff, quantiles, prior1.mixture.wt=prior1.mixture.wt, n2.n1.ratio=n2.n1.ratio,
              compute.prob=T, target=prob, n.start=n.start, n.max=n.max)
  
    out <- c(out, list(quantiles=quantiles, accepted.diff=accepted.diff, prior1=prior1, prior2=prior2, prior1.mixture.wt=prior1.mixture.wt, 
                       prob=prob, n2.n1.ratio=n2.n1.ratio, n.start=n.start, n.max=n.max, exact.results=T))
  }
  else
  {
    out <- .ss.cons.2norm.q(quantiles, accepted.diff, prior1, prior2,
                                                                     sim.size, n.start, n.max, prior1.mixture.wt=prior1.mixture.wt,
                                                                     return.prob=T, target=prob, n2.n1.ratio=n2.n1.ratio, mcs=max.cons.steps.same.dir)

    out <- c(out, list(quantiles=quantiles, accepted.diff=accepted.diff, prior1=prior1, prior2=prior2, prior1.mixture.wt=prior1.mixture.wt, 
                       prob=prob, n2.n1.ratio=n2.n1.ratio, n.start=n.start, n.max=n.max, sim.size=sim.size,
                       max.cons.steps.same.dir=max.cons.steps.same.dir, exact.results=F))
  }
  
  return(out)
} # end of ss.cons.2norm.prob.q


ss.cons.2norm.prob.q.mymarg <- function(quantiles, accepted.diff, prior1, prior2, clinical.prior,
                                                 prob=0.5, n2.n1.ratio=1, n.start=1000, n.max=100000, sim.size=50000,
                                                 max.cons.steps.same.dir=3)
{
  # prior1 and prior2 must be lists with dimensions mu0, n0, prec.shape and prec.rate
  tmp <- .ss.cons.2norm.check.list(prior1, "prior1")
  tmp <- .ss.cons.2norm.check.list(prior2, "prior2")
  tmp <- .ss.cons.2norm.check.list(clinical.prior, "clinical.prior")

  tmp <- .ss.cons.2norm.known.sds(prior1, prior2)
  
  if (all(tmp$known))
  {
    out <- .ss.cons.2norm.q.known.sds(tmp, accepted.diff, quantiles, clinical.prior=clinical.prior, n2.n1.ratio=n2.n1.ratio,
              compute.prob=T, target=prob, n.start=n.start, n.max=n.max)
  
    out <- c(out, list(quantiles=quantiles, accepted.diff=accepted.diff, prior1=prior1, prior2=prior2, clinical.prior=clinical.prior, 
                       prob=prob, n2.n1.ratio=n2.n1.ratio, n.start=n.start, n.max=n.max, exact.results=T))
  }
  else
  {
    out <- .ss.cons.2norm.q(quantiles, accepted.diff, prior1, prior2,
                                                                     sim.size, n.start, n.max,
                                                                     clinical.prior=clinical.prior, return.prob=T, target=prob, n2.n1.ratio=n2.n1.ratio, mcs=max.cons.steps.same.dir)

    out <- c(out, list(quantiles=quantiles, accepted.diff=accepted.diff, prior1=prior1, prior2=prior2, clinical.prior=clinical.prior, 
                       prob=prob, n2.n1.ratio=n2.n1.ratio, n.start=n.start, n.max=n.max, 
                       sim.size=sim.size, max.cons.steps.same.dir=max.cons.steps.same.dir, exact.results=F))
  }
  
  return(out)
} # end of ss.cons.2norm.prob.q.mymarg


ss.cons.2norm.prob.q.bothmarg <- function(quantiles, accepted.diff, prior1, prior2,
                                                 prob=0.5, n2.n1.ratio=1, n.start=1000, n.max=100000, sim.size=50000,
                                                 max.cons.steps.same.dir=3)
{
  # prior1 and prior2 must be lists with dimensions mu0, n0, prec.shape and prec.rate
  tmp <- .ss.cons.2norm.check.list(prior1, "prior1")
  tmp <- .ss.cons.2norm.check.list(prior2, "prior2")

  tmp <- .ss.cons.2norm.known.sds(prior1, prior2)
  
  if (all(tmp$known))
  {
    out <- .ss.cons.2norm.q.known.sds(tmp, accepted.diff, quantiles, n2.n1.ratio=n2.n1.ratio,
              compute.prob=T, target=prob, both=T, n.start=n.start, n.max=n.max)
  
    out <- c(out, list(quantiles=quantiles, accepted.diff=accepted.diff, prior1=prior1, prior2=prior2, prob=prob,
                       n2.n1.ratio=n2.n1.ratio, n.start=n.start, n.max=n.max, exact.results=T))
  }
  else
  {
    out <- .ss.cons.2norm.q(quantiles, accepted.diff, prior1, prior2,
                                                                     sim.size, n.start, n.max,
                                                                     return.prob=T, both=T, target=prob, n2.n1.ratio=n2.n1.ratio, mcs=max.cons.steps.same.dir)

    out <- c(out, list(quantiles=quantiles, accepted.diff=accepted.diff, prior1=prior1, prior2=prior2, prob=prob,
                       n2.n1.ratio=n2.n1.ratio, n.start=n.start, n.max=n.max, 
                       sim.size=sim.size, max.cons.steps.same.dir=max.cons.steps.same.dir, exact.results=F))
  }
  
  return(out)
} # end of ss.cons.2norm.prob.q.bothmarg


ss.plot <- function(ss.out, zero=0.1, show.fit=T)
{
  log.n <- log(pmax(zero, ss.out$n.visited))
  n.plot <- pmax(exp(zero), ss.out$n.visited)
  out <- ss.out$outcome
  plot(n.plot, out, xlab='Sample Size', ylab='Outcome', xlog=T)

  if (show.fit)
  {
    df <- data.frame(out=out, log.n=log.n, log.n2 = log.n^2)
    glm.out <- glm(out~log.n+log.n2, data=df)
    log.n <- log(pmax(zero, par('usr')[c(1,2)]))
    log.n <- seq(from=log.n[1], to=log.n[2], length=300)
    df <- data.frame(log.n=log.n, log.n2=log.n^2)
    predict.out <- predict(glm.out, df)
    points(exp(log.n), predict.out, type='l', col='orange')
  }
} # end of ss.plot


# :::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
# ::: Definition of internal functions, not to be called directly by user                           :::::
# ::: (will be called by other functions with proper arguments)                                     :::::


.ss.cons.2norm.PostMoments.sample <- function(n, sim.size, prior1, prior2, clinical.prior, prior1.mixture.wt=0.5)
{
  # n: length 2
    
  gen.data <- function(sim.size, n, prior)
  {
    # n: length 2
    
    sds <- .ss.conserencebetween2means.known.sds.summary(prior)
    
    # First group
        
    if (sds$known[1])
    {
      lambda <- 1/(sds$value[1]^2)
    }
    else lambda <- rgamma(sim.size, prior$prec.shape[1], prior$prec.rate[1])
    
    
    if (n[1] > 1)
    {
      ns2.1 <- rgamma(sim.size, (n[1]-1)/2, lambda/2)
    }
    else ns2.1 <- rep(0, sim.size)
    
    mu <- rnorm(sim.size, prior$mu0[1], sd=1/sqrt(prior$n0[1]*lambda))
    
    if (n[1] > 0)
    {
      xbar.1 <- rnorm(sim.size, mu, sd=1/sqrt(n[1]*lambda))
    }
    else
    {
      xbar.1 <- rep(0, sim.size)
    }
    
    # Second group
    
 
    if (!sds$common) 
    {
      if (sds$known[2])
      {
        lambda <- 1/(sds$value[2]^2)
      }
      else lambda <- rgamma(sim.size, prior$prec.shape[2], prior$prec.rate[2])
    }
    
    if (n[2] > 1)
    {
      ns2.2 <- rgamma(sim.size, (n[2]-1)/2, lambda/2)
    }
    else ns2.2 <- rep(0, sim.size)
    
    mu <- rnorm(sim.size, prior$mu0[2], sd=1/sqrt(prior$n0[2]*lambda))
    
    if (n[2] > 0)
    {
      xbar.2 <- rnorm(sim.size, mu, sd=1/sqrt(n[2]*lambda))
    }
    else
    {
      xbar.2 <- rep(0, sim.size)
    }
    
    # Join two group values
    xbar <- matrix(c(xbar.1, xbar.2), nrow=2, byrow=T)
    ns2  <- matrix(c(ns2.1,  ns2.2),  nrow=2, byrow=T)
    
    list(xbar=xbar, ns2=ns2)
  } # end of gen.data
  

  # Posterior moments for mu, from one specialist point of view
  post.moments <- function(n, prior, xSample)
  {
    # Following results from pp. 216-219, Joseph & Belisle, 1997
    
    mu <- (n*xSample$xbar + prior$n0 * prior$mu0) / (n + prior$n0) 
    mu <- as.vector(matrix(c(-1,1), nrow=1)%*%mu) 
    
    sds <- .ss.conserencebetween2means.known.sds.summary(prior)   
  
    if (sds$common & all(!sds$known))
    {
      # common but unknown variances (t distrn)
      
      J <- matrix(c(1,1), nrow=1, ncol=2)
      n.factor <- matrix(n*prior$n0/(n+prior$n0), nrow=1, ncol=2)
      m <- n + prior$n0
      
      B <- 2*prior$prec.rate + as.vector(J%*%xSample$ns2) + as.vector(n.factor%*%((xSample$xbar-prior$mu0)^2))
      C <- mean(n) + prior$prec.shape
      D <- prod(m)/sum(m)
      
      t.df <- 2*C
      t.scale.factor <- sqrt(B/(2*C*D))
      var <- NULL
    }
    else
    {
      # (approx) Normal distrn
      
      var <- matrix(NA, nrow=length(mu), ncol=2)
      
      for (g in 1:2)
      {
        if (!sds$known[g])
        {
          beta.n <- prior$prec.rate[g] + xSample$ns2[g,]/2 + n[g]*prior$n0[g]/2/(n[g]+prior$n0[g])*(xSample$xbar[g,]-prior$mu0[g])^2
          var[,g] <- 2*beta.n/(n[g]+prior$n0[g])/(2*prior$prec.shape[g]+n[g]-2)
        }
        else var[,g] <- (sds$value[g]^2)/(n[g]+prior$n0[g])
      }

      J <- matrix(c(1,1), nrow=2, ncol=1)
      var <- as.vector(var%*%J)
      t.scale.factor <- NULL
      t.df <- NULL
    }

    list(mu=mu, var=var, t.df=t.df, t.scale.factor=t.scale.factor)
  } # end of post.moments
  
  
  # Generate (xbar, ns2) from clinical.prior or from a mixture of the two specialists prior distrns
  
  if (.ss.cons.defined.list(clinical.prior))
  {
    xSample <- gen.data(sim.size, n, clinical.prior)
  }
  else
  {
    sim.size1 <- rbinom(1, sim.size, prior1.mixture.wt)
    sim.size2 <- sim.size - sim.size1 
    sample1 <- gen.data(sim.size1, n, prior1)
    sample2 <- gen.data(sim.size2, n, prior2)
    xSample <- mapply(cbind, sample1, sample2, SIMPLIFY=F)
  }
  
      
  # Compute posterior moments for means difference from each specialist point of view

  pm1 <- post.moments(n, prior1, xSample)
  pm2 <- post.moments(n, prior2, xSample)
  
  list(specialist1=pm1, specialist2=pm2)
} # end of .ss.cons.2norm.PostMoments.sample


.ss.cons.2norm.hpdlimits <- function(accepted.diff, prior1, prior2, 
                                               level, sim.size, 
                                               n.start, n.max, next.n=.ss.bsearch,
                                               clinical.prior=list(), prior1.mixture.wt=0.5,
                                               target, both=F, return.prob=F, n2.n1.ratio=1, mcs=3, min.for.possible.return=2^ceiling(1.5*mcs))
{  
  increasing.outcome.with.n <- return.prob
        
  # Initial values

  n1 <- min(n.start, n.max)
  step0 <- max(ceiling(n1/100), min.for.possible.return) # Initial step
  outcome <- numeric(0)
  n.visited <- numeric(0)
  
  ncol <- ifelse(both, 2, 1)
  
  out <- list(continue=T, target=target, increasing.outcome.with.n=increasing.outcome.with.n, step0=step0, n.max=n.max, mcs=mcs, min.for.possible.return=min.for.possible.return)


  while (out$continue)
  {
    n <- ceiling(n1*c(1, n2.n1.ratio))
    
    if (both)
    {
      post.moments1 <- .ss.cons.2norm.PostMoments.sample(n, sim.size, prior1, prior2, prior1)
      post.moments2 <- .ss.cons.2norm.PostMoments.sample(n, sim.size, prior1, prior2, prior2)
      specialist1 <- mapply(c, post.moments1$specialist1, post.moments2$specialist1, SIMPLIFY=F)
      specialist2 <- mapply(c, post.moments1$specialist2, post.moments2$specialist2, SIMPLIFY=F)
      post.moments <- list(specialist1=specialist1, specialist2=specialist2)
    }
    else
    {
      post.moments <- .ss.cons.2norm.PostMoments.sample(n, sim.size, prior1, prior2, clinical.prior, prior1.mixture.wt=prior1.mixture.wt)
    }
    
    # Compute HPD intervals for means difference, for each specialist
    
    if (!is.null(post.moments$specialist1$t.df))
    {
      t.stat <- qt((1+level)/2, post.moments$specialist1$t.df)
      specialist1.hpd.lower <- post.moments$specialist1$mu - post.moments$specialist1$t.scale.factor*t.stat
      specialist1.hpd.upper <- post.moments$specialist1$mu + post.moments$specialist1$t.scale.factor*t.stat 
    }
    else
    {
      z.stat <- qnorm((1+level)/2)
      specialist1.hpd.lower <- post.moments$specialist1$mu - sqrt(post.moments$specialist1$var)*z.stat
      specialist1.hpd.upper <- post.moments$specialist1$mu + sqrt(post.moments$specialist1$var)*z.stat 
    }
    
    
    if (!is.null(post.moments$specialist2$t.df))
    {
      t.stat <- qt((1+level)/2, post.moments$specialist1$t.df)
      specialist2.hpd.lower <- post.moments$specialist2$mu - post.moments$specialist2$t.scale.factor*t.stat
      specialist2.hpd.upper <- post.moments$specialist2$mu + post.moments$specialist2$t.scale.factor*t.stat 
    }
    else
    {
      z.stat <- qnorm((1+level)/2)
      specialist2.hpd.lower <- post.moments$specialist2$mu - sqrt(post.moments$specialist2$var)*z.stat
      specialist2.hpd.upper <- post.moments$specialist2$mu + sqrt(post.moments$specialist2$var)*z.stat 
    }
    
    
    # Compute difference between the two specialists HPD interval limits
    out.diff <- abs(matrix(c(specialist2.hpd.lower-specialist1.hpd.lower, specialist2.hpd.upper-specialist1.hpd.upper), ncol=2))
    out.diff <- pmax(out.diff[,1], out.diff[,2])
    out.diff <- matrix(out.diff, ncol=ncol)    
 
 
    if (return.prob) out.diff <- out.diff <= accepted.diff
    J <- matrix(1, nrow=1, ncol=sim.size)
    out.diff <- J%*%out.diff
    out.diff <- out.diff/sim.size
    
    if (both) out.diff <- ifelse(return.prob, min(out.diff), max(out.diff))    
    tmp <- as.vector(out.diff)

    # ------------------------------  

    outcome <- c(tmp, outcome)
    n.visited <- c(n1, n.visited)

    out$outcome <- outcome
    out$n.visited <- n.visited

    nextn <- next.n(out)
    out$continue <- nextn$continue
    n1 <- nextn$n
  }
  
  n <- ceiling(n1*c(1, n2.n1.ratio))

  list(n=n, n.visited=rev(out$n.visited), outcome=rev(out$outcome))
} # end of .ss.cons.2norm.hpdlimits


.ss.cons.2norm.cdf <- function(cdf.points, accepted.cdf.diff, prior1, prior2, 
                                               sim.size, n.start, n.max, next.n=.ss.bsearch, 
                                               clinical.prior=list(), 
                                               prior1.mixture.wt=0.5,
                                               target=accepted.cdf.diff, both=F, return.prob=F, n2.n1.ratio=1, mcs=3, min.for.possible.return=2^ceiling(1.5*mcs))
{  
  specialist.estimates <- function(cdf.points.rep, posterior.moments)
  {
    if (!is.null(posterior.moments$t.df))
    {
      out <- pt((cdf.points.rep - posterior.moments$mu)/posterior.moments$t.scale.factor, posterior.moments$t.df) 
    }
    else
    {
      out <- pnorm((cdf.points.rep - posterior.moments$mu)/sqrt(posterior.moments$var)) 
    }
    
    out
  } # end of specialist.estimates
  
  
  increasing.outcome.with.n <- return.prob
        
  # Initial values

  n1 <- min(n.start, n.max)
  step0 <- max(ceiling(n1/100), min.for.possible.return) # Initial step
  outcome <- numeric(0)
  n.visited <- numeric(0)

  ncol <- ifelse(both, 2, 1)
  
  out <- list(continue=T, target=target, increasing.outcome.with.n=increasing.outcome.with.n, step0=step0, n.max=n.max, mcs=mcs, min.for.possible.return=min.for.possible.return)

  m.cdf.points <- length(cdf.points)
  if (m.cdf.points == 1) cdf.points.rep <- cdf.points
  else cdf.points.rep <- rep(cdf.points, rep(ncol*sim.size, m.cdf.points))


  while (out$continue)
  {
    n <- ceiling(n1*c(1, n2.n1.ratio))
    
    if (both)
    {
      post.moments1 <- .ss.cons.2norm.PostMoments.sample(n, sim.size, prior1, prior2, prior1)
      post.moments2 <- .ss.cons.2norm.PostMoments.sample(n, sim.size, prior1, prior2, prior2)
      specialist1 <- mapply(c, post.moments1$specialist1, post.moments2$specialist1, SIMPLIFY=F)
      specialist2 <- mapply(c, post.moments1$specialist2, post.moments2$specialist2, SIMPLIFY=F)
      post.moments <- list(specialist1=specialist1, specialist2=specialist2)
    }
    else
    {
      post.moments <- .ss.cons.2norm.PostMoments.sample(n, sim.size, prior1, prior2, clinical.prior, prior1.mixture.wt=prior1.mixture.wt)
    }
    

    # Compute cumulative probabilities for means difference, for each specialist
    
    specialist1.points <- specialist.estimates(cdf.points.rep, post.moments$specialist1)
    specialist2.points <- specialist.estimates(cdf.points.rep, post.moments$specialist2)  
        
    # Compute difference between the two specialists cumulative probabilities

    out.diff <- abs(specialist2.points-specialist1.points)
    out.diff <- matrix(out.diff, ncol=m.cdf.points)
    out.diff <- apply(out.diff, 1, max)
    out.diff <- matrix(out.diff, ncol=ncol)
 
    if (return.prob) out.diff <- out.diff <= accepted.cdf.diff
    J <- matrix(1, nrow=1, ncol=sim.size)
    out.diff <- J%*%out.diff
    out.diff <- out.diff/sim.size
    
    if (both) out.diff <- ifelse(return.prob, min(out.diff), max(out.diff))    
    tmp <- as.vector(out.diff)

    # ------------------------------  

    outcome <- c(tmp, outcome)
    n.visited <- c(n1, n.visited)

    out$outcome <- outcome
    out$n.visited <- n.visited

    nextn <- next.n(out)
    out$continue <- nextn$continue
    n1 <- nextn$n
  }
  
  n <- ceiling(n1*c(1, n2.n1.ratio))

  list(n=n, n.visited=rev(out$n.visited), outcome=rev(out$outcome))
} # end of .ss.cons.2norm.cdf


.ss.cons.2norm.q <- function(quantiles, accepted.diff, prior1, prior2,
                                               sim.size, n.start, n.max, next.n=.ss.bsearch,
                                               clinical.prior=list(),
                                               prior1.mixture.wt=0.5,
                                               target, both=F, return.prob=F, n2.n1.ratio=1, mcs=3, min.for.possible.return=2^ceiling(1.5*mcs))
{
  specialist.estimates <- function(quantiles, posterior.moments)
  {
    if (!is.null(post.moments$specialist1$t.df))
    {
      out <- posterior.moments$mu + posterior.moments$t.scale.factor * qt(quantiles, posterior.moments$t.df)
    }
    else
    {
      out <- posterior.moments$mu + sqrt(posterior.moments$var) * qnorm(quantiles)
    }
    
    out
  } # end of specialist.estimates
  

  increasing.outcome.with.n <- return.prob

  # Initial values

  n1 <- min(n.start, n.max)
  step0 <- max(ceiling(n1/100), min.for.possible.return) # Initial step
  outcome <- numeric(0)
  n.visited <- numeric(0)

  ncol <- ifelse(both, 2, 1)

  out <- list(continue=T, target=target, increasing.outcome.with.n=increasing.outcome.with.n, step0=step0, n.max=n.max, mcs=mcs, min.for.possible.return=min.for.possible.return)

  m.quantiles <- length(quantiles)
  if (m.quantiles == 1) quantiles.rep <- quantiles
  else quantiles.rep <- rep(quantiles, rep(ncol*sim.size, m.quantiles))


  while (out$continue)
  {
    n <- ceiling(n1*c(1, n2.n1.ratio))

    if (both)
    {
      post.moments1 <- .ss.cons.2norm.PostMoments.sample(n, sim.size, prior1, prior2, prior1)
      post.moments2 <- .ss.cons.2norm.PostMoments.sample(n, sim.size, prior1, prior2, prior2)
      specialist1 <- mapply(c, post.moments1$specialist1, post.moments2$specialist1, SIMPLIFY=F)
      specialist2 <- mapply(c, post.moments1$specialist2, post.moments2$specialist2, SIMPLIFY=F)
      post.moments <- list(specialist1=specialist1, specialist2=specialist2)
    }
    else
    {
      post.moments <- .ss.cons.2norm.PostMoments.sample(n, sim.size, prior1, prior2, clinical.prior, prior1.mixture.wt=prior1.mixture.wt)
    }

    # Compute quantiles for means difference, for each specialist
    
    specialist1.points <- specialist.estimates(quantiles.rep, post.moments$specialist1)
    specialist2.points <- specialist.estimates(quantiles.rep, post.moments$specialist2)
    
    # Compute difference between the two specialists quantiles
    
    out.diff <- abs(specialist2.points-specialist1.points)
    out.diff <- matrix(out.diff, ncol=m.quantiles)
    out.diff <- apply(out.diff, 1, max)
    out.diff <- matrix(out.diff, ncol=ncol)

    if (return.prob) out.diff <- out.diff <= accepted.diff
    J <- matrix(1, nrow=1, ncol=sim.size)
    out.diff <- J%*%out.diff
    out.diff <- out.diff/sim.size

    if (both) out.diff <- ifelse(return.prob, min(out.diff), max(out.diff))
    tmp <- as.vector(out.diff)

    # ------------------------------

    outcome <- c(tmp, outcome)
    n.visited <- c(n1, n.visited)

    out$outcome <- outcome
    out$n.visited <- n.visited

    nextn <- next.n(out)
    out$continue <- nextn$continue
    n1 <- nextn$n
  }

  n <- ceiling(n1*c(1, n2.n1.ratio))

  list(n=n, n.visited=rev(out$n.visited), outcome=rev(out$outcome))
} # end of .ss.cons.2norm.q


.ss.bsearch <- function(mylist)
{
  ##################################################################
  # mylist: consists in a list of the following arguments:
  # -------
  #
  # n.visited
  # outcome
  # n.max
  # mcs
  # min.for.possible.return
  # target
  # step0
  # increasing.outcome.with.n  
  #
  ##################################################################

  all.same.dir <- function(nvisited){length(unique(sign(diff(nvisited)))) == 1}

  #

  last.outcome <- mylist$outcome[1]
  dir <- ifelse(last.outcome==mylist$target, -1,
                ifelse(mylist$increasing.outcome.with.n, -1, 1)*sign(last.outcome-mylist$target))
  continue <- T

  if (length(mylist$n.visited) == 1)
  {
    n <- ifelse(dir == 1, mylist$n.visited + dir * mylist$step0, floor(mylist$n.visited/2))
    n <- min(max(0, n), mylist$n.max)
  }
  else
  {
    steps <- -diff(mylist$n.visited)

    if (abs(steps[1]) == 1 && length(unique(sign(mylist$outcome[1:2]-mylist$target))) == 2)
    {
      cond <- ifelse(rep(mylist$increasing.outcome.with.n, 2),
                     mylist$outcome[1:2] >= mylist$target, mylist$outcome[1:2] <= mylist$target)
      n <- min(mylist$n.visited[1:2][cond])
      continue <- F
    }
    else if (length(mylist$n.visited) > mylist$mcs && all(mylist$n.visited[1:(mylist$mcs+1)]==mylist$n.max) && dir == 1)
    {
      n <- mylist$n.max
      continue <- F
    }
    else if (length(mylist$n.visited) > mylist$mcs && all(mylist$n.visited[1:(mylist$mcs+1)]==0) && dir == -1)
    {
      n <- 0
      continue <- F
    }
    else if (mylist$n.visited[1] == max(mylist$n.visited) && dir == 1 && abs(steps[1]) != 1)
    {
      n <- mylist$n.visited[1] + 2*max(abs(diff(mylist$n.visited)))
      n <- min(n, mylist$n.max)
    }
    else if (mylist$n.visited[1] == min(mylist$n.visited) && dir == -1 && abs(steps[1]) != 1)
    {
      n <- floor(mylist$n.visited[1]/2)
    }
    else
    {
      possible.to.move.back <- abs(steps[1]) > mylist$min.for.possible.return

      if (possible.to.move.back && length(mylist$n.visited) > mylist$mcs && all.same.dir(mylist$n.visited[1:(mylist$mcs+1)]) && sign(steps[1]) == dir && 
          (length(mylist$n.visited) == (mylist$mcs+1) || !all.same.dir(mylist$n.visited[1:(mylist$mcs+2)])) )
      {
        n <- mylist$n.visited[mylist$mcs+2]
      }
      else
      {
        if (abs(steps[1]) == 1)
        {
          m <- mylist$n.visited[1] + dir
        }
        else
        {
          d <- mylist$n.visited - mylist$n.visited[1]
          m <- mylist$n.visited[sign(d)==dir][1]
          m <- ceiling((mylist$n.visited[1]+m)/2)
        }

        if (dir == 1)
        {
          n <- max(m, mylist$n.visited[1]+1)
        }
        else
        {
          n <- min(m, mylist$n.visited[1]-1)
        }

        n <- min(n, mylist$n.max)
        n <- max(0, n)
      }
    }
  }

  # If continue=F, then n indicates the optimal sample size found

  list(n=n, continue=continue)
} # end of .ss.bsearch


.ss.cons.2norm.check.list <- function(test.prior, prior.name)
{
  if (missing(test.prior))
  {
    stop(paste(c(prior.name, " list undefined."), collapse=''), call.=F) 
  }
  else
  {
    tmp <- try(paste(test.prior), silent=T)
    if (class(tmp) != "try-error")
    {
      if (is.null(test.prior$mu0)) stop(paste(c("$mu0 must be a dimension (of length 2) of ", prior.name, " list."), collapse=''), call.=F)
      if (length(test.prior$mu0) != 2) stop(paste(c("$mu0 must be of length 2 in ", prior.name, " list."), collapse=''), call.=F)
      
      if (is.null(test.prior$n0)) stop(paste(c("$n0 must be a dimension (of length 2) of ", prior.name, " list."), collapse=''), call.=F)
      if (length(test.prior$n0) != 2) stop(paste(c("$n0 must be of length 2 in ", prior.name, " list."), collapse=''), call.=F)

      any.known.sd <- ifelse(is.null(test.prior$known.sd), F, any(!is.na(test.prior$known.sd)))
      common.sd  <- ifelse(is.null(test.prior$common.sd), F, test.prior$common.sd)
      shape.len  <- ifelse(is.null(test.prior$prec.shape), 0, length(test.prior$prec.shape))
      rate.len   <- ifelse(is.null(test.prior$prec.rate),  0, length(test.prior$prec.rate))
      
      if (!any.known.sd)
      {
        if (common.sd)
        {
          if (shape.len != 1) stop(paste(c("$prec.shape must be of length 1 when $common.sd is TRUE in ", prior.name, " list."), collapse=''), call.=F)
          if (rate.len != 1)  stop(paste(c("$prec.rate must be of length 1 when $common.sd is TRUE in ", prior.name, " list."), collapse=''), call.=F)
        }
        else
        {
          if (shape.len != 2) stop(paste(c("$prec.shape must be of length 2 when $common.sd is FALSE [or undefined] in ", prior.name, " list."), collapse=''), call.=F)
          if (rate.len != 2)  stop(paste(c("$prec.rate must be of length 2 when $common.sd is FALSE [or undefined] in ", prior.name, " list."), collapse=''), call.=F)
        }
      }
      else
      {
        # any known sd
        first.sd.is.known  <- !is.na(test.prior$known.sd[1])
        second.sd.is.known <- !is.na(test.prior$known.sd[2])
        
        if (!second.sd.is.known)
        {
          if (!common.sd)
          {
            if (shape.len != 2) stop(paste(c("$prec.shape must be of length 2 when $common.sd is FALSE [or undefined] in ", prior.name, " list."), collapse=''), call.=F)
            if (rate.len != 2)  stop(paste(c("$prec.rate must be of length 2 when $common.sd is FALSE [or undefined] in ", prior.name, " list."), collapse=''), call.=F)
          }
        }
        else
        {
          # second sd is known
          
          if (first.sd.is.known)
          {
            # known sd = T T
            
            if (common.sd)
            {
              if (length(unique(test.prior$known.sd)) != 1) stop(paste(c("Values in $known.sd must be identical when $common.sd is TRUE in ", prior.name, " list."), collapse=''), call.=F) 
            }
          }
          else
          {
            # known sd = F T
            
            if (common.sd)
            {
              stop(paste(c("If variance is really common in both groups ($common.sd is TRUE), then $known.sd must either be undefined or defined, but with the same value in each of its two components (in ", prior.name, " list)."), collapse=''), call.=F)
            }
            else
            {
              if (shape.len != 1) stop(paste(c("$prec.shape must be of length 1 when $common.sd is FALSE [or undefined] and sd in 2nd group is known in ", prior.name, " list."), collapse=''), call.=F)
              if (rate.len != 1)  stop(paste(c("$prec.rate must be of length 1 when $common.sd is FALSE [or undefined] and sd in 2nd group is known in ", prior.name, " list."), collapse=''), call.=F)
            }
          }
        }
      }
    }
    else
    {
      stop(paste(c(prior.name, " list undefined."), collapse=''), call.=F)
    }
  }
} # end of .ss.cons.2norm.check.list


.ss.cons.2norm.known.sds <- function(prior1, prior2)
{
  specialist.knows.both.sds <- function(prior)
  {
    known <- F
    any.known.sd <- ifelse(is.null(prior$known.sd), F, any(!is.na(prior$known.sd[seq(2)])))
    
    if (any.known.sd)
    {
      common.sd <- ifelse(is.null(prior$common.sd), F, prior$common.sd)
      sds <- prior$known.sd[seq(2)]
      
      if (common.sd)
      {
        known <- T
        sds <- rep(sds[!is.na(sds)], 2)
      }
      else
      {
        known <- all(!is.na(sds))
      }
    }  
    
    # Prepare object to return  
    
    out <- list(known=known)
    if (known)
    {
      out$sds <- sds
      out$mu0 <- prior$mu0
      out$n0 <- prior$n0
    }
    
    out
  } # end of specialist.knows.both.sds


  tmp1 <- specialist.knows.both.sds(prior1)
  tmp2 <- specialist.knows.both.sds(prior2)
  
  if (tmp1$known & tmp2$known)
  {
    out <- mapply(c, tmp1, tmp2)
    out$sds <- matrix(out$sds, nrow=2)
    out$mu0 <- matrix(out$mu0, nrow=2)
    out$n0 <- matrix(out$n0, nrow=2)
  }
  else
  {
    out <- list(known=F)
  }

  out
} # end of .ss.cons.2norm.known.sds


.ss.cons.absNorm.mean <- function(mu, prec, prior1.mixture.wt=numeric(0), both=logical(0))
{
  # Returns E|N(mu,prec)|
  # mu and prec can be vectors of length 2

  pi <- 4 * atan(1)
  mean <- sqrt(2/(pi*prec))*exp(-prec*(mu^2)/2) + mu*(2*pnorm(sqrt(prec)*mu)-1)
  mean <- ifelse(!both & length(mu)==2, sum(c(prior1.mixture.wt,1-prior1.mixture.wt)*mean), max(mean))
  mean
} # end of .ss.cons.absNorm.mean


.ss.cons.absNorm.cdf <- function(b, mu, prec, r, d, prior1.mixture.wt=numeric(0), both=logical(0))
{
  # Return P{b*|N(mu,prec)| + |r| <= d}
  # mu and prec can be vectors of length 2
  # b is assumed positive

  d <- d - abs(r)
  mu.len <- length(mu)
  
  if (d < 0)
  {
    prob <- rep(0, mu.len)
  }
  else
  {
    ncp <- prec*(mu^2)
    q <- prec*((d/b)^2)
    prob <- pchisq(q, 1, ncp=ncp)
  }
  
  prob <- ifelse(!both & mu.len==2, sum(c(prior1.mixture.wt,1-prior1.mixture.wt)*prob), min(prob))
  prob
} # end of .ss.cons.absNorm.cdf


.ss.cons.2norm.hpdlimits.known.sds <- function(known.sds.list, accepted.diff, level=0.95, prior1.mixture.wt=0, n2.n1.ratio=1,
              n.start=1000, n.max=100000, target=accepted.diff, next.n=.ss.nextn4nonrndoutcomes, both=F, compute.prob=F,
              clinical.prior=list())
{
  h <- function(n, known.sds.list, z, both=logical(0),
                     return.prob=F, accepted.diff=numeric(0),
                     prior1.mixture.wt=numeric(0), clinical.prior=list())
  {
    m <- .ss.conserencebetween2means.known.sds.parms(n, known.sds.list, clinical.prior=clinical.prior)
    abs.Cn <- z * abs(diff(m$sigmank))  


    if (return.prob)
    {
      res <- .ss.cons.absNorm.cdf(1, m$Yn.mean + m$delta.an, 1/m$Yn.var, abs.Cn, accepted.diff, prior1.mixture.wt=prior1.mixture.wt, both=both)
    }
    else
    {
      res <- abs.Cn + .ss.cons.absNorm.mean(m$Yn.mean + m$delta.an, 1/m$Yn.var, prior1.mixture.wt=prior1.mixture.wt, both=both)
    }
    
    res
  } # end of h
  
  
  z <- qnorm((1+level)/2)
  
  # Initial values

  n1 <- min(n.start, n.max)
  step0 <- max(ceiling(n1/100), 10) # Initial step
  outcome <- numeric(0)
  n.visited <- numeric(0)

  out <- list(continue=T, target=target, increasing.outcome.with.n=compute.prob, step0=step0, n.max=n.max)
 
  while (out$continue)
  {
    n <- ceiling(n1*c(1, n2.n1.ratio))
    h.out <- h(n, known.sds.list, z, both=both, prior1.mixture.wt=prior1.mixture.wt, clinical.prior=clinical.prior, return.prob=compute.prob, accepted.diff=accepted.diff)

    outcome <- c(h.out, outcome)
    n.visited <- c(n1, n.visited)

    out$outcome <- outcome
    out$n.visited <- n.visited

    nextn <- next.n(out)
    out$continue <- nextn$continue
    n1 <- nextn$n
  }  

  list(n=n1, n.visited=rev(out$n.visited), outcome=rev(out$outcome))

} # end of .ss.cons.2norm.hpdlimits.known.sds


.ss.cons.2norm.cdf.known.sds <- function(known.sds.list, cdf.points, accepted.cdf.diff, prior1.mixture.wt=0, n2.n1.ratio=1,
              n.start=1000, n.max=100000, target=accepted.cdf.diff, next.n=.ss.nextn4nonrndoutcomes, both=F, clinical.prior=list())
{
  ratio.of.two.correlated.normal.distrns.cdf <- function(w, mu.num, mu.denom, sigma.num, sigma.denom, rho)
  {
    # Function (very) slightly modified from
    # "The R-code for computing the cdf and the df of the ratio of two correlated Normal rv"
    # De Capitani and Pollastri

    # w: can be of length 2  -> return  P(w[1] < Y/X < w[2])
    #                  or 1  -> return  P(Y/X < w)

    # definition of the quantities in expression (2)
    b <- mu.denom/sigma.denom
    a   <- sqrt(1/(1-rho^2)) * (mu.num/sigma.num-rho*b)
    t.w <- sqrt(1/(1-rho^2)) * (sigma.denom/sigma.num*w-rho)

    # introduction of auxialiry variables
    A <- (a-b*t.w)/sqrt(1+t.w^2)
    B <- -b
    C <- t.w/sqrt(1+t.w^2)

    #definition of the two addends in expression (1)

    res <- numeric(0)
    for (i in seq(along=w))
    {
      Mcor <- matrix(c(1,C[i],C[i],1),2,2)
      A1 <- pmvnorm(lower=c(A[i],B),upper=Inf, corr=Mcor)
      A2 <- pmvnorm(lower=c(-A[i],-B),upper=Inf, corr=Mcor)
      attributes(A1) <- NULL; attributes(A2) <- NULL
      res[i] <- A1 + A2
    }
  
    if (length(res) == 2) res <- abs(diff(res))
    return(res)
  } # end of ratio.of.two.correlated.normal.distrns.cdf
  
  
  pn <- function(n, known.sds.list, cdf.points, both=F, accepted.cdf.diff=numeric(0), 
                     prior1.mixture.wt=numeric(0), clinical.prior=list())
  {
    m <- .ss.conserencebetween2means.known.sds.parms(n, known.sds.list, clinical.prior=clinical.prior)
    
    unk <- (cdf.points - m$ank)/m$sigmank # length: 2
    vnk <- -m$bnk/m$sigmank[col(m$bnk)] # dim: 2 x 2
    
    delta.un <- diff(unk)   # scalar
    un.dot.bar <- mean(unk) # scalar
    delta.vn <- as.vector(vnk%*%t(m$D))     # length: 2
    vn.dot.bar <- as.vector(vnk%*%t(m$J))/2 # length: 2
    
    corr.num <- -as.vector(m$J%*%(delta.vn*vn.dot.bar*m$xbar.var))                                       # length: 1|2
    corr.denom <- as.vector(sqrt(m$J%*%(m$xbar.var*vn.dot.bar^2)) * sqrt(m$J%*%(m$xbar.var*delta.vn^2))) # length: 1|2 
    corr <- corr.num/corr.denom                                                                          # length: 1|2
    
    num.mean <- -un.dot.bar - as.vector(m$J%*%(vn.dot.bar*m$mu0s)) # length: 1|2
    num.var <- 1 + as.vector(m$J%*%(m$xbar.var*vn.dot.bar^2))      # length: 1|2
    denom.mean <- delta.un + as.vector(m$J%*%(delta.vn*m$mu0s))    # length: 1|2
    denom.var <- as.vector(m$J%*%(m$xbar.var*delta.vn^2))          # length: 1|2
 
    w <- c(-1/2, 1/2)
    res <- numeric(ncol(mu0s))
    for (g in seq(along=res)) res[g] <- ratio.of.two.correlated.normal.distrns.cdf(w, num.mean[g], denom.mean[g], sqrt(num.var[g]), sqrt(denom.var[g]), corr[g])
    if (length(res) == 2) res <- ifelse(both, min(res), sum(res*c(prior1.mixture.wt, 1-prior1.mixture.wt)))
    
    res
  } # end of pn
  
  
  if (is.na(match("mvtnorm", rownames(installed.packages()))))
  {
    options(CRAN="http://cran.us.r-project.org")
    install.packages("mvtnorm")
  }
  if (is.na(match("mvtnorm", (.packages())))) library(mvtnorm)
  
  
  # Initial values

  n1 <- min(n.start, n.max)
  step0 <- max(ceiling(n1/100), 10) # Initial step
  outcome <- numeric(0)
  n.visited <- numeric(0)
  cdf.points <- sort(cdf.points)

  out <- list(continue=T, target=target, increasing.outcome.with.n=F, step0=step0, n.max=n.max)
 
  while (out$continue)
  {
    n <- ceiling(n1*c(1, n2.n1.ratio))
    pn.out <- pn(n, known.sds.list, cdf.points, both=both, prior1.mixture.wt=prior1.mixture.wt, clinical.prior=clinical.prior, accepted.cdf.diff=accepted.cdf.diff)

    outcome <- c(pn.out, outcome)
    n.visited <- c(n1, n.visited)

    out$outcome <- outcome
    out$n.visited <- n.visited

    nextn <- next.n(out)
    out$continue <- nextn$continue
    n1 <- nextn$n
  }  

  list(n=n1, n.visited=rev(out$n.visited), outcome=rev(out$outcome))

} # end of .ss.cons.2norm.cdf.known.sds


.ss.cons.2norm.q.known.sds <- function(known.sds.list, accepted.diff, quantiles, prior1.mixture.wt=0, n2.n1.ratio=1,
              n.start=1000, n.max=100000, target=accepted.diff, next.n=.ss.nextn4nonrndoutcomes, both=F, compute.prob=F,
              clinical.prior=list())
{
  q <- function(n, known.sds.list, z, both=logical(0),
                     return.prob=F, accepted.diff=numeric(0),
                     prior1.mixture.wt=numeric(0), clinical.prior=list())
  {
    m <- .ss.conserencebetween2means.known.sds.parms(n, known.sds.list, clinical.prior=clinical.prior)    
    Dn <- m$delta.an + z*diff(m$sigmank)
  
    
    if (return.prob)
    {
      res <- .ss.cons.absNorm.cdf(1, m$Yn.mean + mean(Dn), 1/m$Yn.var, abs(diff(Dn))/2, accepted.diff, prior1.mixture.wt=prior1.mixture.wt, both=both)
    }
    else
    {
      res <- abs(diff(Dn))/2 + .ss.cons.absNorm.mean(m$Yn.mean + mean(Dn), 1/m$Yn.var, prior1.mixture.wt=prior1.mixture.wt, both=both)
    }
    
    res
  } # end of q
  
  
  z <- unique(qnorm(c(min(quantiles), max(quantiles))))
  
  # Initial values

  n1 <- min(n.start, n.max)
  step0 <- max(ceiling(n1/100), 10) # Initial step
  outcome <- numeric(0)
  n.visited <- numeric(0)

  out <- list(continue=T, target=target, increasing.outcome.with.n=compute.prob, step0=step0, n.max=n.max)
 
  while (out$continue)
  {
    n <- ceiling(n1*c(1, n2.n1.ratio))
    q.out <- q(n, known.sds.list, z, both=both, prior1.mixture.wt=prior1.mixture.wt, clinical.prior=clinical.prior, return.prob=compute.prob, accepted.diff=accepted.diff)

    outcome <- c(q.out, outcome)
    n.visited <- c(n1, n.visited)

    out$outcome <- outcome
    out$n.visited <- n.visited

    nextn <- next.n(out)
    out$continue <- nextn$continue
    n1 <- nextn$n
  }  

  list(n=n1, n.visited=rev(out$n.visited), outcome=rev(out$outcome))

} # end of .ss.cons.2norm.q.known.sds


.ss.conserencebetween2means.known.sds.summary <- function(prior)
{
  common <- ifelse(is.null(prior$common.sd), F, prior$common.sd)
        
  if (is.null(prior$known.sd))
  {
    known <- rep(F, 2)
    value <- rep(NA, 2)
  }
  else 
  {
    known <- !is.na(prior$known.sd)
    if (length(known) == 1) known[2] <- known & common
    value <- prior$known.sd[1:2]
      
    if (common & any(known))
    {
      known <- rep(T, 2)
      if (length(prior$known.sd) != 2 || any(is.na(value)))
      {
        value <- prior$known.sd[1:2]
        value <- value[!is.na(value)]
        value <- rep(value, 2)
      }
    }
  }
    
  list(common=common, known=known, value=value)
} # end of .ss.conserencebetween2means.known.sds.summary 


.ss.nextn4nonrndoutcomes <- function(mylist)
{
  ##################################################################
  # mylist: consists in a list of the following arguments:
  # -------
  #
  # n.visited
  # outcome
  # target
  # step0
  # increasing.outcome.with.n
  # n.max  
  #
  ##################################################################

  continue <- T

  sufficient.n <- (mylist$increasing.outcome.with.n & mylist$outcome >= mylist$target) | (!mylist$increasing.outcome.with.n & mylist$outcome <= mylist$target)
  n.visited <- mylist$n.visited
  
  if (all(sufficient.n))
  {
    n.visited <- sort(n.visited)
    
    if (n.visited[1] == 0)
    {
      n <- 0
      continue <- F
    }
    else
    {
      if (length(n.visited) == 1)
      {
        n <- max(n.visited - mylist$step0, 0)
      }
      else
      {
        n <- max(n.visited[1] - 2*diff(n.visited[1:2]), 0)
      }
    }
  }
  else if (all(!sufficient.n))
  {
    n.visited <- rev(sort(n.visited))
    
    if (n.visited[1] == mylist$n.max)
    {
      n <- Inf
      continue <- F
    }
    else
    {
      if (length(n.visited) == 1)
      {
        n <- n.visited + mylist$step0
      }
      else
      {
        n <- n.visited[1] + 2 * (n.visited[1]-n.visited[2])
      }
    
      n <- min(n, mylist$n.max)
    }
  }
  else
  {
    smallest.sufficient.n  <- min(n.visited[sufficient.n])
    largest.insufficient.n <- max(n.visited[!sufficient.n])
    
    if ((smallest.sufficient.n-largest.insufficient.n) == 1)
    {
      n <- smallest.sufficient.n
      continue <- F
    }
    else
    {
      n <- floor((smallest.sufficient.n + largest.insufficient.n)/2)
    }
  }

  # If continue=F, then n indicates the optimal sample size found

  list(n=n, continue=continue)
} # end of .ss.nextn4nonrndoutcomes


.ss.cons.lines.intersection <- function(a, b)
{
  # a, b: vectors of length 2
  
  x <- -diff(a)/diff(b)
  y <- a[1] + b[1] * x
  
  list(x=x, y=y)
} # end of .ss.cons.lines.intersection


.ss.conserencebetween2means.known.sds.parms <- function(n, known.sds.list, clinical.prior=list())
{
  D <- matrix(c(-1,1), nrow=1)
  J <- matrix(c(+1,1), nrow=1)
  
  ank.elem <- known.sds.list$n0*known.sds.list$mu0 / (n + known.sds.list$n0)
  ank <- as.vector(D%*%ank.elem) # length: 2
  delta.an <- diff(ank)
    
  sigma2nk.elem <- known.sds.list$sds^2 / (n + known.sds.list$n0)
  sigmank <- sqrt(as.vector(J%*%sigma2nk.elem)) # length: 2
    
  bnk <- c(-1,1) * n / (n + known.sds.list$n0) # dim: 2 x 2
  delta.bn <- as.vector(bnk %*% t(D)) # length: 2
    

  if (!.ss.cons.defined.list(clinical.prior))
  {
    # use a mixture of the two specialists priors as clinical prior
    mu0s <- known.sds.list$mu0
    xbar.var <- known.sds.list$sds^2 * (1/n + 1/known.sds.list$n0) # 2 x 2
  }
  else
  {
    # use specified clinical.prior
    mu0s <- matrix(clinical.prior$mu0, ncol=1)
      
    sds <- .ss.cons.2norm.known.sds.summary(clinical.prior)
    xbar.var <- 1/n + 1/clinical.prior$n0 # to be multiplied (below)
      
    for (g in 1:2)
    {
      if (sds$known[g])
      {
        xbar.var[g] <- xbar.var[g] * (sds$value[g]^2)
      }
      else
      {
        xbar.var[g] <- xbar.var[g] * clinical.prior$prec.rate[g] / (clinical.prior$prec.shape[g] - 1)
      }  
    }
      
    xbar.var <- matrix(xbar.var, ncol=1) # dim: 2 x 1
  }
  
  Yn.mean <- as.vector(J %*% (mu0s * delta.bn))  # length: 2
  Yn.var <- as.vector(J%*%(delta.bn^2*xbar.var)) # length: 1|2
    
  list(D=D, J=J, ank=ank, bnk=bnk, delta.an=delta.an, delta.bn=delta.bn, mu0s=mu0s, sigmank=sigmank, xbar.var=xbar.var, Yn.mean=Yn.mean, Yn.var=Yn.var)
} # end of .ss.conserencebetween2means.known.sds.parms


.ss.cons.defined.list <- function(mylist){ifelse(length(mylist) == 0, F, any(unlist(lapply(mylist, length)) > 0))}
