c**************************CONTENTS OF search.f************************
c
c BRACKN       !Bracket n such that f(n_1).f(n_2)<0, for n in [n_1,n_2]
c FINDN                        !Search n such that f(n)<=0 and f(n-1)>0
c SEARCHLEN                                          !Search HPD length
c SEARCHINT                                     !Search best HPD length
c
c**********************************************************************
c
c  Given beta parameters a,b
c        credible set CS=1-alpha
c        interval length 0<w<1
c     Compute:
c      HPD length   given coverage CS or
c      HPD coverage given length   w
c
c INPUT :
c met : ACC=1, WOC(AC)=2, ALC=3, WOC(AL)=4
c       NOTE - only WOC(AC) is computed 
c CS   is HPD coverage
c w    is HPD length
c a, b are prior beta parameters
c nl   is lower sample size and nu is upper sample size
c fnl  is for function associated to nl
c bracket is true for sample size n in [nl,nu], otherwise false.
c 
c OUTPUT :
c updated lower sample size bound       nl
c updated upper sample size bound       nu
c updated function fnl associated to    nl
c bracketed sample size true/false      bracket 
c
c**********************************************************************
c
      SUBROUTINE BRACKN(long,met,CS,w,a,b,nl,nu,fnl,bracket)

c     ..Parameters..
      integer          MAXIT,izero,ione,itwo,addn
      double precision zero,half
      parameter       (MAXIT=20,izero=0,ione=1,itwo=2,
     *                 addn=10,zero=0.0d0,half=0.5d0)

c     ..Arguments..
      integer          met,nl,nu
      double precision CS,w,a,b,fnl
      logical          long,bracket

c     ..Local..
      integer          typ,nbr,i
      double precision fnbr,fnu,cnt
      character*17     bracknn,bracknf

c     ..Calling procedures
      double precision BAYSMP                  !Posterior probability
c     SRCHINFO                                  !Current work summary

c BASIC INITIALIZATION

      bracket = .true.                        !n is assumed bracketed
      nbr     = nu                           !frequentist sample size
      nu      = izero
      nl      = nu
      fnu     = zero
      fnl     = fnu
      i       = ione                                !iteration number
      typ     = ione                            !counter for printing
      bracknn = 'BRACKN:  nl  nu ='
      bracknf = 'BRACKN: fnl fnu ='

c METHOD ADOPTION

      if (met.le.itwo) then
       cnt = CS                       !Average Coverage, fixed length
      else
       cnt = w                        !Average Length, fixed coverage
      endif

c CASE: INITIAL SAMPLE SIZE > 0

      if (nbr.gt.izero) then

       fnbr = BAYSMP(met,CS,w,a,b,nbr) - cnt
       if (met.le.itwo) fnbr = -fnbr                          !for AC

       if (fnbr.le.zero) then            !initial sample size too big

        nu  = nbr                              !new upper sample size
        fnu = fnbr                    !new upper sample size function
  100   nl  = int(nu*half)                     !new lower sample size
        fnl = BAYSMP(met,CS,w,a,b,nl) - cnt
        if (met.le.itwo) fnl = -fnl                           !for AC
        call SRCHINFO(long,i,bracknn,bracknf,nl,nu,fnl,fnu,typ)

        if (fnl.gt.zero) GOTO 500              !sample size bracketed
        i   = i + ione
        nu  = nl                      !decrease new upper sample size
        fnu = fnl                        !decrease new upper function 
        typ = izero
        if ((nl.le.izero).or.(i.gt.MAXIT)) GOTO 400          !nu=nl=0
        GOTO 100

       else                          !initial sampsize func too small

        nl  = nbr                              !new lower sample size
        fnl = fnbr                    !new lower sample size function
  200   nu  = int(nl*itwo)                     !new upper sample size
        fnu = BAYSMP(met,CS,w,a,b,nu) - cnt
        if (met.le.itwo) fnu = -fnu                           !for AC
        call SRCHINFO(long,i,bracknn,bracknf,nl,nu,fnl,fnu,typ)

        if (fnu.le.zero) GOTO 500              !sample size bracketed
        i = i + ione
        if (i.gt.MAXIT)  GOTO 400
        nl  = nu                      !increase new upper sample size
        fnl = fnu                        !increase new upper function
        typ = izero
        GOTO 200

       endif                       !end initial sampsize func too big

      endif                            !end initial positive sampsize

c CASE: INITIAL SAMPLE SIZE == 0

      if (nbr.eq.izero) then

       fnbr = BAYSMP(met,CS,w,a,b,nbr) - cnt
       if (met.le.itwo) fnbr = -fnbr                          !for AC

       if (fnbr.le.zero) GOTO 400                     !n(initial) = 0
       nl  = nbr                               !new lower sample size
       fnl = fnbr                     !new lower sample size function
  300  nu  = nl + addn
       fnu = BAYSMP(met,CS,w,a,b,nu) - cnt
       if (met.le.itwo) fnu = -fnu                            !for AC
       call SRCHINFO(long,i,brackn,nl,nu,fnl,fnu,typ)

       if (fnu.le.zero)  GOTO 500              !sample size bracketed
       i = i + ione
       if (i.ge.MAXIT)   GOTO 400
       nl   = nu
       fnl  = fnu
       typ  = izero
       GOTO 300

      endif                                   !end initial sampsize=0

c CASE: INITIAL SAMPLE SIZE < 0 (ERROR)

      if (nbr.lt.izero) GOTO 400

c ALL IS CLEAR

      GOTO 500

 400  bracket=.false.
      call SRCHINFO(long,i,bracknn,bracknf,nl,nu,fnl,fnu,typ)
      if (i.gt.MAXIT)  
     * write(6,'(8x,a,1x,i3,1x,a)')            'are not bracketed in',
     *                                              i-1,'iterations.'

 500  RETURN
      END
c
c************************************************************************
c
c Find n such that f(na)<=0 and f(nb)>0, where
c n is said to be bracketed, i.e. n \in [nl,nu] (bisection)
c INPUT:
c  long is for short or detailed output (defunct)
c  met 1   : g(n)   >= 0 and g(n-1) < 0 (increasing function: AC)
c  met 3   : g(n-1) >= 0 and g(n)   < 0 (decreasing function: AL)
c  met 2   : WO for AC ; met 4 : WO for AL (defunct)
c  lower and upper sample size na, nb ; fna is f(na)
c  HPD coverage ; HPD length w ; a,b beta parameters
c OUTPUT:
c  error message err 
c  final sample size nfin
c
c************************************************************************
c
      subroutine FINDN(long,met,na,nb,fna,CS,w,a,b,err,nfin)

c     ..Parameters..
      integer          MAXIT,MINUM,izero,ione,itwo,ithree
      double precision half,zero
      parameter        (ione=1,itwo=2,ithree=3,MAXIT=50,
     *                  MINUM=itwo,izero=0,half=0.5d0,zero=0.0d0)

c     ..Arguments..
      integer          met,na,nb,nfin
      double precision w,a,b,CS,fna
      logical          long,err

c     ..Local..
      integer          np,i
      double precision fnp,cnt
      character*3      mtype
      character*17     findnn,findnf

c     ..calling functions..
      double precision BAYSMP                  !Posterior probability
c     SRCHINFO                                  !Current work summary
c     COVER                                          !min cov max len

c BASIC INITIALIZATION

      nfin   = izero                         !final sample size (n=0)
      i      = ione                                 !iteration number
      findnn = 'FINDN:  nl  nu  ='
      findnf = 'FINDN: fnl fnu  ='
      
c METHOD ADOPTION

      if (met.le.itwo) then
       cnt = CS                       !Average Coverage, fixed length
       if (met.eq.ione) then
        mtype = 'ACC'
       else
        mtype = 'WOC'
       endif
      else
       cnt = w                        !Average Length, fixed coverage
       if (met.eq.ithree) then
        mtype = 'ALC'
       else
        mtype = 'WOC'
       endif
      endif

c BEGIN SAMPLE SIZE SEARCH

  100 np  = na + int((nb-na)*half)            !begin bisection search
      fnp = BAYSMP(met,CS,w,a,b,np) - cnt
      if (met.le.itwo) fnp = -fnp                             !for AC

      call SRCHINFO(long,i,findnn,findnf,na,np,fna,fnp,izero)

      if ((nb-na).le.MINUM) then

       if ((fna*fnp).lt.zero) then
        nfin = np
       else
        nfin = nb
       endif
       write(6,'(//26x,2(a,1x),i10)')   mtype,'sample size =',nfin
       call COVER(long,met,CS,w,a,b,nfin)

      elseif (((fna*fnp).lt.zero).and.((np-na).le.ione)) then

       nfin = np
       write(6,'(//26x,2(a,1x),i10)')   mtype,'sample size =',nfin
       call COVER(long,met,CS,w,a,b,nfin)

      else                                           !end step search

       i = i + ione
       if (i.gt.MAXIT)  GOTO 200
       if ((fna*fnp).ge.zero) then
        na  = np
        fna = fnp
       else
        nb  = np
       endif
       GOTO 100

      endif                                     !end bisection search

c  ALL IS CLEAR

      GOTO 300

  200 err = .true.
      write(6,'(/6x,a,1x,i3,1x,a)')        'FINDN : no root found in',
     *                                             MAXIT,'iterations.'

  300 RETURN
      END
c
c************************************************************************
c
c Given initial length guess lenit at x, search
c  for HPD length len at lenx. If down = .true. search
c   for decreasing length, else search for increasing length
c
c************************************************************************
c
      SUBROUTINE SEARCHLEN(MIN,MAX,TOL,down,a,b,
     *                     CS,len1,lenit,x,xf,xf1,len,lenx)

c     ..Parameters..
      double precision zero,incr
      parameter       (zero=0.0d0,incr=10.0d0)

c     ..Arguments..
      double precision MIN,MAX,TOL,incr,a,b,CS,
     *                 len1,lenit,x,xf,xf1,len,lenx
      logical          down

c     ..Local..
      double precision mult

c     ..Calling procedures..
c     COVLEN                                          !HPD coverage
c     SEARCHINT               !search for best interval [lena,lenb]

      mult = MIN
      len1 = lenit
  100 if (down) then
       len1 = len1 - mult
       if (len1.lt.MIN) len1 = MIN
      else
       len1 = len1 + mult
       if (len1.gt.MAX) len1 = MAX
      endif

c      ..get HPD coverage xf1 at x given Beta parms a,b, length len1

      call COVLEN(1,a,b,len1,CS,x,xf1)
      xf1  = xf1 - CS

      if ((xf*xf1).le.zero) then !function bracketed
       call SEARCHINT(len1,lenit,a,b,TOL,CS,xf1,len,lenx)
      elseif (mult.gt.MAX) then
       down = .false.
      else
       mult = mult*incr
       GOTO 100
      endif

      RETURN
      END
c
c************************************************************************
c
c Bisection search of best HPD length w (called from SEARCHLEN)
c Find a solution such that fa.fb < 0
c  f = cdfbeta(x+w,a,b)-cdfbeta(x,a,b)-CS
c INPUT:
c   f continuous on the interval [lena,lenb], f(a).f(b) < 0
c OUTPUT 
c   approximate length len at lenxx or failure message
c
c************************************************************************
c
      SUBROUTINE SEARCHINT(lena,lenb,a,b,TOL,CS,flena,len,lenx)

c     ..Parameters..
      integer          MAXIT
      double precision zero,half
      parameter       (MAXIT=100,zero=0.0d0,half=0.50d0)

c     ..Arguments..
      double precision lena,lenb,a,b,TOL,CS,flena,len,lenx

c     ..Local..
      integer          i
      double precision flenb

c     ..Calling procedures..
c     COVLEN                                             !HPD coverage

      do 100 i = 1,MAXIT

       len   = lena + (lenb - lena)*half

c      ..get HPD coverage flenb at lenx given Beta parms a,b, lengh len

       call COVLEN(1,a,b,len,CS,lenx,flenb)
       flenb = flenb - CS

       if((abs(flenb).le.TOL).or.((lenb-lena)*half.le.TOL)) GOTO 200
       if ((flena*flenb).ge.zero) then
        lena  = len
        flena = flenb
       else
        lenb  = len
       endif

  100 continue

  200 if (i.gt.MAXIT) then
       write(6,'(6X,A,i3,A)')    'SEARCHINT : no root found in',
     *                                      MAXIT,' iterations'
       STOP 'in SEARCHINT: no root'
      endif

      RETURN
      END
c
c************************************************************************
c
