c***********************CONTENTS OF util.f******************************
c
c   COVLEN                             !HPD coverage, HPD/non HPD length
c   EQTLEN                                !Equal-tailed (non HPD) length
c   HPDLEN                                                   !HPD length
c   HPDCOV                                                 !HPD coverage
c   INVBETA                             !Inverse cdf beta(a,b) a>=1,b>=1
c   NRBETA            !Simple Newton-Raphson to improve accuracy INVBETA
c   FINDX                               !Call to Modified Newton-Raphson
c   RTSAFE      !Modified Newton-Raphson to find x such that f(x)=f(x+w)
c   FGR                         !External for RTSAFE (called from FINDX)
c   CDF                       !External for RTSAFE (called from INVBETA)
c   GAMMLN                                                 !Log of Gamma
c   CDFBETA                       !Beta Cumulative distribution function
c   BETACF                        !Continued fraction method for CDFBETA
c   PDFBETA                                    !Probability density Beta
c
c
c***********************************************************************
c
c  calls to compute HPD coverage, HPD length or non HPD length
c  if arg = 1 get HPD coverage len at lenx given beta parameters a,b
c                 and length w.
c  if arg = 2 get non HPD length len at lenx given beta parameters
c                 a,b, and coverage CS;
c             get HPD length len1 at lenx1 given initial non HPD 
c                 length len, beta parameters a,b and coverage CS.
c
c***********************************************************************
c
      SUBROUTINE COVLEN(arg,a,b,w,CS,lenx,len,lenx1,len1)

c      ..Arguments..
       integer          arg
       double precision a,b,w,CS,lenx,len,lenx1,len1

c      ..Local..
       double precision tmp

       if (arg.eq.1) then
        call HPDCOV(a,b,w,lenx,len)
       else
        tmp = 0.0d0
        call EQTLEN(a,b,CS,lenx,len)
        tmp = len
        call HPDLEN(tmp,a,b,CS,lenx1,len1)
       endif

      RETURN
      END
c
c***********************************************************************
c
c     ..Compute equal-tailed (non HPD) length len
c     ..len=I^{-1}[1-pl,a,b]-I^{-1}[pl,a,b]
c     ..I is inverse of beta cdf
c     INPUT  : beta parameters a,b, ; coverage probability CS;
c     OUTPUT : lower probability pl, non HPD length len
c
c***********************************************************************
c
      SUBROUTINE EQTLEN(a,b,CS,pl,len)

c     ..Parameters..
      double precision half,one
      parameter        (half=0.50d0,one=1.0d0)

c     ..Arguments..
      double precision a,b,CS,pl,len

c     ..Calling procedures
      double precision INVBETA                   !Inverse of Beta dist'n

      pl  = half*(one - CS)                     !lower probability bound
      len = INVBETA(one - pl,a,b)                 !upper bound to length
      pl  = INVBETA(pl,a,b)                       !lower bound to length
      len = len - pl                           !length based on symmetry

      RETURN
      END
c
c***********************************************************************
c 
c Given initial guess lenit, compute by Modified Newton-Raphson
c  HPD interval length len of HPD region, starting from
c  value lenx.
c INPUT :
c   initial length lenit, beta parameters a,b,
c   Coverage CS.
c OUTPUT :
c   HPD Interval length len, starting position of len: lenx.
c
c***********************************************************************
c
      SUBROUTINE HPDLEN(lenit,a,b,CS,lenx,len)

c     ..Parameters..
      double precision TOL,MIN,MAX,zero,one
      parameter       (TOL=1.0d-13,MIN=1.0d-05,MAX=1.0d0-MIN,
     *                 zero=0.0d0,one=1.0d0)

c     ..Arguments..
      double precision lenit,a,b,CS,len,lenx

c     ..Local..
      double precision len1,x,xf,xf1
      logical          down

c     ..Temporary Local..
c      double precision mult

c     ..Calling procedures..
c     COVLEN                                     !HPD coverage given 
c     SEARCHLEN                      !search for best interval [a,b]


c      ..get HPD coverage xf at x given Beta parameters a,b, length lenit

      call COVLEN(1,a,b,lenit,CS,x,xf)

      xf  = xf - CS
      if (abs(xf).le.TOL) then
       len = lenit
       lenx = x
       RETURN
      endif

c     ..Given initial guess lenit at x
c     ..begin length search by decreasing size, since
c     ..we expect final length < given length

      down = .true.
      call SEARCHLEN(MIN,MAX,TOL,down,a,b,
     *               CS,len1,lenit,x,xf,xf1,len,lenx)

      if (.not.down) 
     * call SEARCHLEN(MIN,MAX,TOL,down,a,b,
     *               CS,len1,lenit,x,xf,xf1,len,lenx)

      RETURN
      END
c
c*********************************************************************
c 
c      ..get HPD coverage px given x and len
c      ..HPD Coverage px where 
c      ..f(x|a,b) = f(x+len|a,b)
c      ..Given x, px = I(x+len|a,b)-I(x|,b)
c      ..I(.) is cdf beta
c      INPUT : beta parameters a,b ; length len
c      OUTPUT: HPD coverage px at point x
c
c*********************************************************************
c
      SUBROUTINE HPDCOV(a,b,len,x,px)

c     ..Arguments..
      double precision a,b,len,x,px

c     ..Calling procedures..
      double precision FINDX     !x such that f(x)=f(x+len) given a,b
      double precision CDFBETA                   !cdf of beta variate

c     ..get interval point x given interval length len,
c     ..beta parameters a,b

      x  = FINDX(len,a,b)

c     ..HPD coverage

      px = CDFBETA(x+len,a,b) - CDFBETA(x,a,b)

      RETURN
      END
c
c*********************************************************************
c
c Find y such that I(y,a,b) = p, where a,b >=1
c I is incomplete beta function ratio (Kotz and Johnson).
c Approximation due to Abramowitz et al. 26.5.22 and
c  accuracy is improved with a few applications of 
c  a simple Newton Raphson procedure (NRBETA). There may
c  be instances when NRBETA will fail. This usually occurs
c  when stringent values are put on desired length or coverage
c  or both (i.e. p->0 or p-> 1. In such cases,
c  use modified Newton-Raphson (from Numerical Recipes (1990)
c  procedure RTSAFE)
c
c*********************************************************************
c
      FUNCTION INVBETA(p,a,b)

c     ..Parameters..
      double precision TOL,zero,half,one,two
      parameter       (TOL=1.0d-13,zero=0.0d0,half=0.50d0,
     *                 one=1.0d0,two=2.0d0)

c     ..Arguments..
      double precision INVBETA,p,a,b

c     ..Local..
      double precision t,lam,ha,hb,h,omeg,yo           !simple NR
      logical          fail
      double precision lb,ub,x                       !modified NR
      logical          bracket
      integer          cnter,tcnter               !print counters
      save             cnter,tcnter
      data             cnter,tcnter/0,500/

c     ..Calling functions..
      double precision INVNOR     !inverse of cdf standard normal
      double precision NRBETA              !simple newton raphson
      double precision RTSAFE            !modified newton raphson
      external         CDF                            !for RTSAFE
c     OUTPUT1                                    !message to wait

c     ..INITIALIZATION

      fail   = .false.

      if ((p.le.zero).or.(p.ge.one)) then
       STOP 'INVBETA: given percentile <=0 or >=1 out of range'
      else                     !simple Newton-Raphson
       t    = INVNOR(p)
       lam  = 0.1666666667d0*t*t - half
       ha   = one/(two*a - one)
       hb   = one/(two*b - one)
       h    = two/(ha + hb)
       omeg = sqrt(h + lam)*t/h -
     1       (lam + 0.83333333d0 - 0.66666667d0/h) *
     2       (hb - ha)
       yo   = a/(a + b*exp(two*omeg))
       INVBETA = NRBETA(p,a,b,yo,fail)

c      ..NEWTON RAPHSON FAILURE

       if (fail) then                    !modified Newton-Raphson

        cnter = cnter + 1
        if (cnter.eq.tcnter) then
         call OUTPUT1(2)
         tcnter = 0
        endif

c     *   write(6,'(4x,i9,2a)') tcnter,'th simple Newton-Raphson',
c     *                     ' failure: call modified version'
        lb = TOL
        ub = one - TOL
        bracket = .true.
        x = RTSAFE(CDF,a,b,p,lb,ub,TOL,bracket)

c       ..For detailed output (ignore)
c        if (cnter.eq.tcnter) then
c         write(6,'(12x,a,l2)') 'Function is bracketed:',bracket
c         tcnter = tcnter + 500
c        endif

        if (.not.bracket) then
         if (a.le.b) then
          x = TOL
         else
          x = one - TOL
         endif
        endif
        INVBETA = x
       endif                             !modified Newton-Raphson

      endif                               !search in valid region

      RETURN
      END
c
c*********************************************************************
c
c Simple Newton Raphson application to improve approximation
c to inverse of Incomplete beta function ratio 
c (only a few iterations should suffice). If this method
c fails, use a more robust procedure
c
c*********************************************************************
c
      FUNCTION NRBETA(p,a,b,yo,fail)

c     ..Parameters..
      integer          ITMAX
      double precision TOL,zero,one
      parameter        (ITMAX=100,TOL=3.0D-9,zero=0.0d0,one=1.0d0)

c     ..Arguments..
      double precision NRBETA,p,a,b,yo
      logical          fail

c     ..Local..
      integer          i
      double precision y

c     ..Calling procedures
      double precision CDFBETA                            !beta cdf
      double precision PDFBETA                            !beta pdf

      do 100 i = 1,ITMAX
       y = yo - (CDFBETA(yo,a,b) - p)/PDFBETA(yo,a,b)

c     ..Method fails if y>1 or y<0

       if ((y.ge.one).or.(y.le.zero)) then
        fail = .true.
        GOTO 1
       endif

       if (abs(y-yo).lt.TOL) GOTO 1

       yo = y

  100 continue

      if (i.gt.ITMAX) fail=.true.

    1 NRBETA = y

      RETURN
      END
c
c***********************************************************************
c
c Find x such that f(x)=f(x+w), given a,b beta parameters.
c
c***********************************************************************
c
      FUNCTION FINDX(w,a,b)

c     ..Parameters..
      double precision TOL,one
      parameter       (TOL=1.0d-13,one=1.0d0)

c     ..Arguments..
      double precision FINDX,w,a,b

c     ..Local..
      double precision lb,ub,x
      logical          bracket

c     ..Calling procedures..
      double precision RTSAFE                !Modified Newton-Raphson
      external         FGR                                !for RTSAFE

      lb = TOL
      ub = one - (w+TOL)
      bracket = .true.
      x = RTSAFE(FGR,w,a,b,lb,ub,TOL,bracket)
      if (.not.bracket) then
       if (a.le.b) then
        x = TOL
       else
        x = one - (w+TOL)
       endif
      endif

      FINDX = x

      RETURN
      END
c
c***********************************************************************
c
c Combination newton-raphson and bisection algorithm to find root
c Source:    Numerical recipes (1990)
c
c***********************************************************************
c
      FUNCTION RTSAFE(FUNCD,w,a,b,x1,x2,xacc,brack)

c     ..Parameter..
      integer          MAXIT
      double precision zero,half
      parameter       (MAXIT=200,zero=0.0d0,half=0.50d0)

c     ..Arguments..
      double precision RTSAFE,FUNCD
      double precision w,a,b,x1,x2,xacc
      logical          brack

c     ..Local..
      integer          j
      double precision fl,df,fh,xl,xh,swap,dxold,dx,f,temp

      call FUNCD(w,a,b,x1,fl,df)
      call FUNCD(w,a,b,x2,fh,df)

      if(fl*fh.ge.zero) then
       brack=.false.
       RETURN
      endif

      if(fl.lt.zero)then
        xl=x1
        xh=x2
      else
        xh=x1
        xl=x2
        swap=fl
        fl=fh
        fh=swap
      endif

      RTSAFE=half*(x1+x2)
      dxold=abs(x2-x1)
      dx=dxold
      call FUNCD(w,a,b,rtsafe,f,df)

      do 11 j=1,MAXIT
        if(((RTSAFE-xh)*df-f)*((RTSAFE-xl)*df-f).ge.zero
     *      .or. abs(2.0d0*f).gt.abs(dxold*df) ) then
          dxold=dx
          dx=half*(xh-xl)
          RTSAFE=xl+dx
          if(xl.eq.RTSAFE) RETURN
        else
          dxold=dx
          dx=f/df
          temp=RTSAFE
          RTSAFE=RTSAFE-dx
          if(temp.eq.RTSAFE) RETURN
        endif
        if(abs(dx).lt.xacc) RETURN
        call FUNCD(w,a,b,RTSAFE,f,df)
        if(f.lt.zero) then
          xl=RTSAFE
          fl=f
        else
          xh=RTSAFE
          fh=f
        endif
  11  continue

      if (j.gt.MAXIT) STOP
     *      'RTSAFE : exceeding maximum iterations'

      RETURN
      END
c
c*********************************************************************
c
c External function for Numerical recipes RTSAFE, 
c declared in procedure FINDX
c f is function, df is derivative
c
c*********************************************************************
c
      SUBROUTINE FGR(w,a,b,x,f,df)

      double precision TOL,one
      parameter       (TOL=1.0d-321,one=1.0d0)

c     ..Arguments..
      double precision w,a,b,x,f,df

c     ..Local..
      double precision xp,term1,term2,logterms

      xp = x + w

      if (x.lt.TOL) write(6,'(2a)') 'Potential problem with',
     *  'RTSAFE call to FGR: x very, very close to zero'
      logterms = log(xp) - log(x)
      term1    = (a - one)*( logterms )
      logterms = log(one - xp) - log(one - x)
      term2    = (b - one)*( logterms )
      f        = term1 + term2

      logterms = log(xp) + log(x)
      term1    = (a - one)*exp( -logterms )
      logterms = log(one - xp) + log(one - x)
      term2    = (b - one)*exp( -logterms )
      df       = -w*(term1 + term2)

      RETURN
      END
c
c
c*********************************************************************
c
c External function for Numerical recipes RTSAFE,
c declared in procedure INVBETA
c f is function, df is derivative
c
c*********************************************************************
c
      SUBROUTINE CDF(a,b,p,x,f,df)

c     ..Arguments..
      double precision a,b,p,x,f,df

c     ..Calling procedures..
      double precision CDFBETA
      double precision PDFBETA

      f  = CDFBETA(x,a,b) - p
      df = PDFBETA(x,a,b)

      RETURN
      END
c
c*********************************************************************
c
c Numerical recipes (1990) : lngamma(xx), xx>0
c
c*********************************************************************
c
      FUNCTION GAMMLN(xx)

c     ..Arguments..
      double precision GAMMLN,xx

c     ..Local..
      integer          j 
      double precision cof(6),stp,half,one,fpf,x,tmp,ser

      DATA cof,stp/76.18009173D0,-86.50532033D0,24.01409822D0,
     *               -1.231739516D0,.120858003D-2,-.536382D-5,
     *               2.50662827465D0/

      DATA half,one,fpf/0.5D0,1.0D0,5.5D0/

      x   = xx - one
      tmp = x + fpf
      tmp = (x+half) * log(tmp) - tmp
      ser = one
      do 11 j=1,6
        x   = x + one
        ser = ser + cof(j)/x
11    continue

      GAMMLN= tmp + log(stp*ser)

      RETURN
      END
c
c*********************************************************************
c
c
c Numerical recipes (1990): Incomplete beta function ratio I
c
c*********************************************************************
c
      FUNCTION CDFBETA(x,a,b)

c     ..Parameters..
      double precision zero,one,two
      parameter        (zero=0.0d0,one=1.0d0,two=2.0d0)

c     ..Arguments..
      double precision CDFBETA,x,a,b

c     ..Local..
      double precision bt

c     ..calling function..
      double precision GAMMLN                  !log of gamma(x)
      double precision BETACF        !continued fraction method

      bt = zero
      if ((x.lt.one).and.(x.gt.zero)) 
     1  bt = exp(GAMMLN(a + b) 
     2      - GAMMLN(a) - GAMMLN(b)
     3      + a * log(x) + b * log(one - x))

      if (x.lt.(a + one)/(a + b + two)) then
       if (bt.gt.zero) then
        CDFBETA = bt*BETACF(x,a,b)/a
       else
        CDFBETA = zero
       endif
      elseif (bt.gt.zero) then
       CDFBETA = one - (bt*BETACF(one - x,b,a))/b
      else
       CDFBETA = one
      endif

      RETURN
      END
c
c*********************************************************************
c
c Numerical recipes (1990): continued fraction evaluation to  
c   compute cdf beta (CDFBETA)
c
c*********************************************************************
c
      FUNCTION BETACF(x,a,b)

c     ..Parameters..
      integer          ITMAX
      double precision EPS,one
      parameter        (ITMAX=200,EPS=3.0D-9,one=1.0d0)

c     ..Arguments..
      double precision BETACF,x,a,b

c     ..Local..
      integer          m
      double precision am,bm,az,qab,qap,qam,bz,em,
     *                 tem,ap,bp,d,app,bpp,aold

      am = one
      bm = one
      az = one
      qab= a+b
      qap= a+one
      qam= a-one
      bz = one-qab*x/qap

      do 10 m = 1,ITMAX
        em  = m
        tem = em + em
        d   = em * (b-m) * x/((qam+tem)*(a+tem))
        ap  = az + d * am
        bp  = bz + d * bm
        d   =-(a+em) * (qab+em) * x/((qap+tem) * (a+tem))
        app = ap + d * az
        bpp = bp + d * bz
        aold= az
        am  = ap / bpp
        bm  = bp / bpp
        az  = app/ bpp
        bz  =one
        if(abs(az-aold).lt.EPS*abs(az)) GOTO 1
   10 continue

      STOP 'BETACF: a or b too big, or ITMAX=200 too small'

    1 BETACF=az
     
      RETURN
      END
c
c********************************************************************
c
c          pdf of beta 
c
c********************************************************************
c
      FUNCTION PDFBETA(yo,a,b)

c     ..Parameters..
      double precision one
      parameter       (one=1.0d0)

c     ..Arguments..
      double precision PDFBETA,yo,a,b

c     ..Calling procedures..
      double precision GAMMLN                     !log of gamma

      PDFBETA = exp( GAMMLN(a + b) - GAMMLN(a) -
     1          GAMMLN(b) + (a - one)*log(yo) +
     2          (b - one)*log(one - yo) )

      RETURN
      END
c
c********************************************************************
c
